<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}
require_once 'wc-am-client.php';

/**
 * WCPP_Frontend
 *
 * PickingPal modifies WooCommerce UI
 *
 * @class       WCPP_Frontend
 * @category    Class
 */
class WCPP_Api_Manager extends WC_AM_Client_2_9_3 {
	const PRO_PRODUCT_ID           = 1513;
	const STANDARD_PRODUCT_ID      = 1360;
	private $license_transient_key = 'wcpp_license_state';
	private $logger;
	private $logger_context = array( 'source' => 'picking-pal-am-client' );

	private static $instance = null;


	public function __construct(
		$file,
		$product_id,
		$software_version,
		$plugin_or_theme,
		$api_url,
		$software_title = '',
		$text_domain = ''
	) {

		add_action(
			'woocommerce_init',
			function () {
				$this->logger = function_exists( 'wc_get_logger' ) ? wc_get_logger() : false;
			}
		);

		$menu = array(
			'page_title' => WCPP()->get_app_name() . ' License',
			'menu_title' => 'PickingPal License',
		);
		parent::__construct(
			$file,
			$product_id,
			$software_version,
			$plugin_or_theme,
			$api_url,
			$software_title,
			$text_domain,
			$menu
		);

		if ( is_admin() ) {

			add_action(
				'admin_init',
				function ( $product_id ) {
					if ( ! $this->is_activated_license() ) {
							add_action( 'admin_notices', array( $this, 'inactive_notice' ) );
					}
				}
			);

			// Clear the cached activation status if woocommerce tool clears transients
			$transient_key = $this->get_license_transient_key();
			add_action(
				'woocommerce_system_status_tool_executed',
				function ( $tool ) use ( $transient_key ) {
					if ( 'clear_transients' == $tool['id'] ) {
						delete_transient( $transient_key );
					}
				},
				10,
				1
			);

			// Clear the cached activation status if the activation option is updated by API manager
			add_action(
				'update_option',
				function ( $option, $old_value, $value ) use ( $transient_key ) {
					if ( strpos( $option, 'wc_am_client_' ) !== false && strpos( $option, '_activated' ) !== false ) {
						if ( $old_value !== $value ) {
							delete_transient( $transient_key );
						}
					}
				},
				10,
				3
			);

			// Clear the cached activation status if the plugin is upgraded
			add_action(
				'upgrader_process_complete',
				function ( $upgrader_object, $options ) use ( $transient_key ) {
					// Check if the updated type is plugins
					if ( $options['action'] === 'update' && $options['type'] === 'plugin' ) {
						foreach ( $options['plugins'] as $plugin ) {
							// Check if the updated plugin is WCPP
							if ( $plugin === plugin_basename( WCPP_PLUGIN_FILE ) ) {
								delete_transient( $transient_key );
							}
						}
					}
				},
				10,
				2
			);
		}
	}

	public static function get() {
		if ( ! self::$instance ) {
			$product_id     = self::get_product_id();
			self::$instance = new self(
				WCPP_PLUGIN_FILE,
				$product_id,
				WCPP()->get_version(),
				'plugin',
				WCPP()->get_website(),
				WCPP()->get_app_name(),
				''
			);
		}
		return self::$instance;
	}

	public function is_activated_license( $live = false ) {

		if ( ! is_admin() ) {
			return true; // Always return true if not in the admin area
		}

		$transient_key = $this->get_license_transient_key();
		if ( $live === false ) {
			$cached_status = get_transient( $transient_key );
			if ( $cached_status !== false && is_array( $cached_status ) && isset( $cached_status['status'] ) ) {
				return (bool) $cached_status['status'];
			}
		}

		$license_status = (bool) $this->get_api_key_status( true );
		set_transient( $transient_key, array( 'status' => $license_status ), SECONDS_IN_DAY );
		return $license_status;
	}



	/**
	 * Generate the default data.
	 */
	public function activation() {
		// Check if we are on a Multisite or not.
		if ( is_multisite() ) {
			$site_ids = get_sites( array( 'fields' => 'ids' ) );

			// Uninstall the plugin for all these sites.
			foreach ( $site_ids as $site_id ) {
				switch_to_blog( $site_id );
				parent::activation();
				restore_current_blog();
			}
		} else {
			parent::activation();
		}
	}

	private function get_license_transient_key() {
		return $this->license_transient_key . '_' . self::get_product_id();
	}

	/**
	 * This is the value assigned to private variable `wc_am_product_id`
	 * when no product id is passed to WC_AM_Client_x
	 *
	 * @return string
	 */
	private function get_product_id_field_name() {
		$identifier = dirname( untrailingslashit( plugin_basename( WCPP_PLUGIN_FILE ) ) );
		$product_id = strtolower(
			str_ireplace(
				array(
					' ',
					'_',
					'&',
					'?',
					'-',
				),
				'_',
				$identifier
			)
		);
		return 'wc_am_product_id_' . $product_id;
	}

	public static function get_product_id() {
		return WCPP()->is_pro() ? self::PRO_PRODUCT_ID : self::STANDARD_PRODUCT_ID;
	}
}
