<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}



/**
 * WCPP_Admin_Tweak_UI
 *
 * PickingPal modifies WooCommerce UI
 *
 * @class       WCPP_admin
 * @category    Class
 */
class WCPP_Admin_Tweak_UI {

	private $plugin_assets_url;

	public function __construct( $settings ) {
		$this->settings          = $settings;
		$this->plugin_assets_url = WCPP()->plugin_url() . '/assets';

		add_action( 'admin_enqueue_scripts', array( $this, 'admin_enqueue_scripts' ), 11 );

		// ORDERS
		// Orders list view
		// hpos compatible
		add_filter( 'bulk_actions-woocommerce_page_wc-orders', array( $this, 'register_bulk_actions' ) );
		add_filter( 'views_woocommerce_page_wc-orders', array( $this, 'woocommerce_views_edit' ), 110 );
		add_filter( 'woocommerce_order_list_table_prepare_items_query_args', array( $this, 'update_order_list_args' ) );
		add_filter( 'woocommerce_order_query_args', array( $this, 'handle_orders_custom_query_var' ), 10, 2 );

		// legacy (not hpos compatible)
		add_filter( 'bulk_actions-edit-shop_order', array( $this, 'register_bulk_actions' ) );
		add_filter( 'views_edit-shop_order', array( $this, 'woocommerce_views_edit' ), 110 );
		add_action( 'pre_get_posts', array( $this, 'modify_orders_list_query' ) );

		// Order detail view
		add_filter(
			'woocommerce_order_actions',
			function ( $actions ) {
				$actions['print_pick_ticket'] = __( 'Print Pick Ticket', 'woocommerce-pickingpal' );
				return $actions;
			}
		);
		add_action(
			'woocommerce_order_action_print_pick_ticket',
			array( $this, 'woocommerce_order_action_print_pick_ticket' )
		);
		add_action( 'load-edit.php', array( $this, 'woocommerce_custom_bulk_action' ) );
		add_action( 'admin_notices', array( $this, 'woocommerce_custom_bulk_admin_notices' ) );
		add_action( 'woocommerce_admin_order_actions_end', array( $this, 'add_actions' ), 110 );

		// PRODUCTS
		// variation fields
		add_action( 'woocommerce_product_after_variable_attributes', array( $this, 'variable_fields' ), 10, 3 );
		add_action( 'woocommerce_save_product_variation', array( $this, 'save_variable_fields' ), 10, 1 );
		// Products fields
		add_action( 'woocommerce_product_options_general_product_data', array( $this, 'product_fields' ) );
		add_action( 'woocommerce_process_product_meta', array( $this, 'save_product_fields' ) );
		// builk actions for variations
		add_action( 'woocommerce_variable_product_bulk_edit_actions', array( $this, 'add_variable_product_bulk_edit_actions' ) );
		add_action( 'woocommerce_bulk_edit_variations', array( $this, 'do_bulk_edit_variations' ), 10, 4 );
		// search in additional product fields
		// add_filter('the_posts', array($this, 'search_pickingpal_fields'));

		add_filter( 'manage_product_posts_columns', array( $this, 'manage_product_posts_columns' ) );
		add_action( 'manage_product_posts_custom_column', array( $this, 'manage_product_posts_custom_column' ), 10, 2 );
	}

	public function manage_product_posts_columns( $columns ) {
		foreach ( WCPP()->settings_handler->get_warehouse_locations( 'only_enabled' ) as $key => $location ) {
			$columns[ $key ] = $location['label'];
		}

		foreach ( WCPP()->settings_handler->get_skus( 'only_enabled' ) as $key => $sku ) {
			$columns[ $key ] = $sku['label'];
		}

		return $columns;
	}

	public function manage_product_posts_custom_column( $column, $post_id ) {
		if ( key_exists( $column, WCPP()->settings_handler->get_skus( 'only_enabled' ) ) ) {
			$sku_str = WCPP()->helper->get_product_additional_sku( $post_id, $column );

			$product        = wc_get_product( $post_id );
			$sku_str_childs = array();
			foreach ( $product->get_children() as $child_id ) {
				$child = wc_get_product( $child_id );
				if ( $child_sku = WCPP()->helper->get_product_additional_sku( $child->get_id(), $column ) ) {
					// $sku_str_childs[] = '<strong>' . $child->get_name() . '</strong>' . '<br>' . $child_sku;
					$sku_str_childs[] = $child_sku;
				}
			}
			$sku_str_childs = array_unique( $sku_str_childs );

			echo $sku_str_childs ? implode( ', ', $sku_str_childs ) : $sku_str;
		}

		if ( key_exists( $column, WCPP()->settings_handler->get_warehouse_locations( 'only_enabled' ) ) ) {
			$warehouse_location_str = WCPP()->helper->get_product_warehouse_location( $post_id, $column );

			$product                       = wc_get_product( $post_id );
			$warehouse_location_str_childs = array();
			foreach ( $product->get_children() as $child_id ) {
				$child = wc_get_product( $child_id );
				if ( $child_location = WCPP()->helper->get_product_warehouse_location( $child->get_id(), $column ) ) {
					// $warehouse_location_str_childs[] = '<strong>' . $child->get_name() . '</strong>' . '<br>' . $child_location;
					$warehouse_location_str_childs[] = $child_location;
				}
			}
			$warehouse_location_str_childs = array_unique( $warehouse_location_str_childs );

			echo $warehouse_location_str_childs ? implode(
				', ',
				$warehouse_location_str_childs
			) : $warehouse_location_str;
		}
	}

	// For orders list

	public function register_bulk_actions( $bulk_actions ) {
		$bulk_actions['print_pick_ticket'] = __( 'Print Pick Tickets', 'woocommerce-pickingpal' );
		return $bulk_actions;
	}

	public function woocommerce_order_action_print_pick_ticket( $order ) {
		$ids      = (array) $order->get_id();
		$sendback = add_query_arg(
			array(
				'pickingpal_action_onload' => 'print_pick_ticket',
			),
			wp_get_referer()
		);
		wp_redirect( $sendback );
		exit();
	}

	public function woocommerce_custom_bulk_action() {
		$wp_list_table = _get_list_table( 'WP_Posts_List_Table' );
		$action        = $wp_list_table->current_action();

		if ( isset( $_REQUEST['post'] ) ) {
			$ids = $_REQUEST['post'];
		} else {
			return;
		}
		switch ( $action ) {
			case 'print_pick_ticket':
				$sendback = wp_nonce_url( admin_url() );
				$sendback = add_query_arg(
					array(
						'method' => 'pickingpal_print_pick_ticket',
						'ids'    => $ids,
					),
					$sendback
				);
				break;

			default:
				return;
		}
		wp_redirect( $sendback );

		exit();
	}

	public function woocommerce_custom_bulk_admin_notices() {

		global $post_type, $pagenow;
		$display_notice = false;
		if ( WCPP()->is_hpos_enabled() ) {
			$page   = isset( $_REQUEST['page'] ) ? $_REQUEST['page'] : null;
			$action = isset( $_REQUEST['action'] ) ? $_REQUEST['action'] : null;
			if ( $page == 'wc-orders' && $action == 'edit' ) {
				$display_notice = true;
			}
		} elseif ( $pagenow == 'edit.php' && $post_type == 'shop_order' && isset( $_REQUEST['print_pick_ticket_msg'] ) ) {
				$display_notice = true;
		}
		if ( $display_notice ) {
			$message = esc_html( $_REQUEST['print_pick_ticket_msg'] );
			if ( $message ) {
				echo "<div class=\"updated\"><p>$message</p></div>";
			}
		}
	}

	// icons in last column
	public function add_actions( $order ) {
		$order_id = $order->get_id(); ?><a class="button print_pick_ticket wc-action-button" target="_blank"
				title="<?php _e( 'Print Pick Ticket', 'woocommerce-pickingpal' ); ?>"
				data-tip="<?php _e( 'Print Pick Ticket', 'woocommerce-pickingpal' ); ?>"
				data-order-id="<?php echo esc_attr( $order_id ); ?>">
				<img src="<?php echo WCPP()->plugin_imgs_url() . '/printer.png'; ?>"
					alt="<?php _e( 'Print Pick Ticket', 'woocommerce-pickingpal' ); ?>"
					style="width:14px; margin: 7px auto;">
		</a>
		<?php
	}

	// For orders tab

	/**
	 * Subscribes to filter `views_edit-shop_order` && `'views_woocommerce_page_wc-orders'`
	 * Adds a tab to the orders list view for `Print Pick Tickets (not-printed)`
	 * Does not show the tab if there are no orders to print
	 *
	 * @param $arr_filter
	 *
	 * @return mixed
	 */
	public function woocommerce_views_edit( $arr_filter ) {

		$order_ids    = wc_get_orders(
			array(
				'return'              => 'ids',
				'type'                => array( 'shop_order' ),
				'status'              => array( 'wc-processing' ),
				'printed_pick_ticket' => false, // custom query var
			)
		);
		$count        = $order_ids ? count( $order_ids ) : 0;
		$status       = 'not-printed';
		$status_field = WCPP()->is_hpos_enabled() ? 'status' : 'post_status';

		if ( isset( $_REQUEST[ $status_field ] ) and $_REQUEST[ $status_field ] == 'not-printed' ) {
			$class = 'current';
		} else {
			$class = '';
		}
		$text = __( 'Print Pick Tickets', 'woocommerce-pickingpal' );

		if ( $count ) {
			$uri = "edit.php?post_status=$status&amp;post_type=shop_order";
			if ( WCPP()->is_hpos_enabled() ) {
				$uri = "admin.php?page=wc-orders&amp;status=$status";
			}
			$arr_filter[ $status ] = "<a href='$uri' class='$class'>$text <span class=\"count\">($count)</span></a>";
		}

		return $arr_filter;
	}

	/**
	 * Adds filter to the order list view  (Legacy, pre HPOS)
	 * Subscribes to action `pre_get_posts`
	 * Handles the orders list view for `Print Pick Tickets (not-printed)` tab.
	 *
	 * @param array $query - Args for WP_Query.
	 * @return void
	 */
	function modify_orders_list_query( $query ) {
		global $pagenow, $typenow;

		// Check if we are in the admin area and on the main query
		if ( is_admin() && 'edit.php' === $pagenow && 'shop_order' === $typenow && $query->is_main_query() ) {
			if ( isset( $_GET['post_status'] ) && 'not-printed' === $_GET['post_status'] ) {
				$query->set( 'post_status', 'wc-processing' );
				$query->set(
					'meta_query',
					array(
						array(
							'key'     => WCPP()->get_meta_key( 'printed_pick_ticket' ),
							'compare' => 'NOT EXISTS',
						),
					)
				);
			}
		}
	}

	/**
	 * Adds filter to the order list view (HPOS)
	 * Subscribes to filter `woocommerce_order_list_table_prepare_items_query_args`
	 * Handles the orders list view for `Print Pick Tickets (not-printed)` tab.
	 *
	 * @param array $args Query Args passed to wc_get_orders
	 *
	 * @return array Updated Query Args
	 */
	public function update_order_list_args( $args ) {
		if ( WCPP()->is_hpos_enabled() ) {
			$status_filter = isset( $args['status'] ) ? $args['status'][0] : null;
			if ( $status_filter == 'not-printed' ) {
				$args['printed_pick_ticket'] = false;
				$args['status']              = array( 'wc-processing' );
			}
		}
		return $args;
	}


	/**
	 * Handle custom query args added to wc_get_orders.
	 * Subscribes to filter `woocommerce_order_query_args`
	 *
	 * @param array $args - Args for WP_Query.
	 * @return array modified $query
	 */
	function handle_orders_custom_query_var( $args ) {
		if ( isset( $args['printed_pick_ticket'] ) ) {
			$printed              = $args['printed_pick_ticket'] ? true : false;
			$args['meta_query'][] = array(
				'key'     => WCPP()->get_meta_key( 'printed_pick_ticket' ),
				'compare' => $printed ? 'EXISTS' : 'NOT EXISTS',
			);
			if ( ! WCPP()->is_hpos_enabled() ) {
				$args['meta_key']     = WCPP()->get_meta_key( 'printed_pick_ticket' );
				$args['meta_compare'] = 'NOT EXISTS';
			}
			unset( $args['printed_pick_ticket'] );
		}
		return $args;
	}

	// For products

	function product_fields() {
		global $post;
		$product = wc_get_product( $post->ID );

		if ( ! $product->is_type( 'variable' ) ) {

			// foreach (get_option(WCPP()->metakey_skus,array()) as $key => $sku) {
			foreach ( WCPP()->settings_handler->get_option( 'metakey_skus' ) as $key => $sku ) {
				if ( ( $sku['enabled'] == 'off' ) ) {
					continue;
				}
				$value = get_post_meta( $post->ID, $key );
				woocommerce_wp_text_input(
					array(
						'id'          => "{$key}",
						'label'       => __( 'Additional SKU', 'woocommerce-pickingpal' ) . " - {$sku['label']}",
						'placeholder' => '',
						'desc_tip'    => 'true',
						'description' => __( "Enter the {$sku['label']} SKU here.", 'woocommerce-pickingpal' ),
						'value'       => isset( $value[0] ) ? $value[0] : '',
					)
				);
			}

			// foreach (get_option(WCPP()->metakey_warehouse_locations,array()) as $key => $location) {
			foreach ( WCPP()->settings_handler->get_option( 'warehouse_locations' ) as $key => $location ) {
				if ( ( $location['enabled'] == 'off' ) ) {
					continue;
				}
				$value = get_post_meta( $post->ID, $key );
				woocommerce_wp_text_input(
					array(
						'id'          => "{$key}",
						'label'       => __( 'Warehouse Location', 'woocommerce-pickingpal' ) . " {$location['label']}",
						'placeholder' => '',
						'desc_tip'    => 'true',
						'description' => __( "Enter the {$location['label']} here.", 'woocommerce-pickingpal' ),
						'value'       => isset( $value[0] ) ? $value[0] : '',
					)
				);
			}
		}
	}

	function save_product_fields( $post_id ) {
		$product = wc_get_product( $post_id );
		// $skus = get_option(WCPP()->metakey_skus,array());
		$skus = WCPP()->settings_handler->get_option( 'metakey_skus' );
		if ( ! $product->is_type( 'variable' ) ) {
			if ( isset( $post_id ) ) {
				// print_r($_POST);
				// die();
				foreach ( $skus as $key => $sku ) {
					if ( ( $sku['enabled'] == 'off' ) ) {
						continue;
					}
					$value = $_POST[ $key ];
					if ( isset( $value ) ) {
						update_post_meta( $post_id, $key, stripslashes( $value ) );
					}
				}
			}
		}

		// $locations = get_option(WCPP()->metakey_warehouse_locations,array());
		$locations = WCPP()->settings_handler->get_option( 'warehouse_locations' );
		if ( ! $product->is_type( 'variable' ) ) {
			if ( isset( $post_id ) ) {
				foreach ( $locations as $key => $location ) {
					if ( ( $location['enabled'] == 'off' ) ) {
						continue;
					}
					$value = $_POST[ $key ];
					if ( isset( $value ) ) {
						update_post_meta( $post_id, $key, stripslashes( $value ) );
					}
				}
			}
		}
	}

	function variable_fields( $loop, $variation_data, $variation ) {

		// foreach (get_option(WCPP()->metakey_skus,array()) as $key => $sku) {
		foreach ( WCPP()->settings_handler->get_option( 'metakey_skus' ) as $key => $sku ) {
			if ( ( $sku['enabled'] == 'off' ) ) {
				continue;
			}

			$value = get_post_meta( $variation->ID, $key );
			?>
			<p class="form-row form-row-first">
				<label><?php _e( 'Additional SKU', 'woocommerce-pickingpal' ); ?>  <?php esc_html( $sku['label'] ); ?> <a class="tips" data-tip="<?php _e( "Enter the {$sku['label']} SKU here.", 'woocommerce-pickingpal' ); ?>" href="#">[?]</a></label>
				<input type="text" size="5"  name="<?php echo esc_attr( "{$key}[$loop]" ); ?>" id="<?php echo esc_attr( "{$key}[$loop]" ); ?>" value="<?php echo isset( $value[0] ) ? esc_attr( $value[0] ) : ''; ?>" />
			</p>
			<?php
		}

		// foreach (get_option(WCPP()->metakey_warehouse_locations,array()) as $key => $location) {
		foreach ( WCPP()->settings_handler->get_option( 'warehouse_locations' ) as $key => $location ) {
			if ( ( $location['enabled'] == 'off' ) ) {
				continue;
			}

			$value = get_post_meta( $variation->ID, $key );
			?>
			<p class="form-row form-row-first">
				<label><?php _e( 'Warehouse Location', 'woocommerce-pickingpal' ); ?> - <?php esc_html( $location['label'] ); ?>
					<a class="tips" data-tip="<?php _e( 'Enter the location here.', 'woocommerce-pickingpal' ); ?>" href="#">[?]</a>
				</label>
				<input type="text" size="5"  name="<?php echo esc_attr( "{$key}[$loop]" ); ?>" id="<?php echo esc_attr( "{$key}[$loop]" ); ?>" value="<?php echo isset( $value[0] ) ? esc_attr( $value[0] ) : ''; ?>" />
			</p>
			<?php
		}
	}

	function save_variable_fields( $post_id ) {
		if ( isset( $_POST['variable_post_id'] ) ) :
			$variable_post_id = (array) $_POST['variable_post_id'];
			// $skus = get_option(WCPP()->metakey_skus,array());
			$skus = WCPP()->settings_handler->get_option( 'metakey_skus' );
			foreach ( $variable_post_id as $i => $variation_id ) {
				$variation_id = (int) $variation_id;

				foreach ( $skus as $key => $sku ) {
					if ( ( $sku['enabled'] == 'off' ) ) {
						continue;
					}
					$value = $_POST[ $key ];
					if ( isset( $value[ $i ] ) ) {
						update_post_meta( $variation_id, $key, stripslashes( $value[ $i ] ) );
					}
				}
			}

			// $locations = get_option(WCPP()->metakey_warehouse_locations,array());
			$locations = WCPP()->settings_handler->get_option( 'warehouse_locations' );
			foreach ( $variable_post_id as $i => $variation_id ) {
				$variation_id = (int) $variation_id;

				foreach ( $locations as $key => $location ) {
					if ( ( $location['enabled'] == 'off' ) ) {
						continue;
					}
					$value = $_POST[ $key ];
					if ( isset( $value[ $i ] ) ) {
						update_post_meta( $variation_id, $key, stripslashes( $value[ $i ] ) );
					}
				}
			}
		endif;
	}

	// Variable bulk actions

	public function add_variable_product_bulk_edit_actions() {
		// $warehouse_locations = get_option(WCPP()->metakey_warehouse_locations,array());
		$warehouse_locations = WCPP()->settings_handler->get_option( 'warehouse_locations' );
		$warehouse_locations = array_filter(
			$warehouse_locations,
			function ( $e ) {
				return $e['enabled'] == 'on';
			}
		);
		if ( count( $warehouse_locations ) > 0 ) {
			wp_enqueue_script( 'variable_product-pickingpal', WCPP()->plugin_js_url() . '/variable-product.js', array(), WCPP_VERSION );
			?>
			<optgroup label="<?php _e( 'PickingPal', 'woocommerce-pickingpal' ); ?>">
				<?php foreach ( $warehouse_locations as $key => $item ) : ?>
				<option data-key="<?php echo esc_attr( $key ); ?>" value="pickingpal_set_warehouse_location"><?php echo __( 'Set Warehouse Location', 'woocommerce-pickingpal' ) . ' - ' . esc_html( $item['label'] ); ?></option>
				<?php endforeach; ?>
			</optgroup>
			<?php
		}
	}

	public function do_bulk_edit_variations( $bulk_action, $data, $product_id, $variations ) {
		if ( $bulk_action === 'pickingpal_set_warehouse_location' ) {
			foreach ( $variations as $variation_id ) {
				update_post_meta( $variation_id, $data['key'], stripslashes( $data['value'] ) );
			}
		}
	}

	// For search in products
	function search_pickingpal_fields( $posts, $query = false ) {
		// $skus = get_option(WCPP()->metakey_skus,array());
		$skus = WCPP()->settings_handler->get_option( 'metakey_skus' );
		if ( ! $skus ) {
			return $posts;
		}
		global $wp_query;

		if ( isset( $wp_query->query['posts_per_page'] ) ) {
			$available_count = $wp_query->query['posts_per_page'] - count( $posts );
		} else {
			$available_count = true;
		}

		if ( is_search() &&
			isset( $wp_query->query['post_type'] ) &&
			in_array( $wp_query->query['post_type'], array( 'product', 'product_variation' ) ) &&
			$available_count
		) {
			$ids = array( 0 );
			foreach ( $posts as $post ) {
				$ids[] = $post->ID;
			}

			$searched_posts = $this->get_posts_search( get_search_query(), $ids, $skus );

			$new_posts = array();
			if ( $searched_posts ) {
				foreach ( $searched_posts as $product_id ) {
					$new_posts[] = get_post( $product_id->post_id );
				}
			}

			$wp_query->found_posts += count( $new_posts );
			if ( $available_count !== true ) {
				$new_posts = array_splice( $new_posts, 0, $available_count );
			}

			return array_merge( $posts, $new_posts );
		}
		return $posts;
	}

	function get_posts_search( $query, $ids, $skus ) {
		global $wpdb, $wp_query;

		$skus_names = array();
		foreach ( $skus as $key => $value ) {
			$skus_names[] = "pm.meta_key='" . $key . "'";
		}
		$skus_names        = implode( ' OR ', $skus_names );
		$bids              = implode( ',', $ids );
		$variable_products = $wpdb->get_results(
			"
          SELECT p.post_parent as post_id FROM $wpdb->posts as p
          INNER JOIN $wpdb->postmeta pm
          ON p.ID = pm.post_id
          AND pm.meta_value LIKE '%$query%'
          INNER JOIN $wpdb->postmeta visibility
          ON p.post_parent = visibility.post_id
          AND visibility.meta_key = '_visibility'
          AND visibility.meta_value <> 'hidden'
          WHERE $skus_names
          AND p.post_parent <> 0
          AND p.ID not in ($bids)
          AND p.post_status = 'publish'
          GROUP BY p.post_parent
          "
		);
		foreach ( $variable_products as $post ) {
			$ids[] = $post->post_id;
		}

		$bids = implode( ',', $ids );

		$simple_products = $wpdb->get_results(
			"SELECT p.ID as post_id FROM $wpdb->posts as p
                INNER JOIN $wpdb->postmeta pm
                ON p.ID = pm.post_id
                AND pm.meta_value LIKE '%$query%'
                INNER JOIN $wpdb->postmeta visibility
                ON p.ID = visibility.post_id
                AND visibility.meta_key = '_visibility'
                AND visibility.meta_value <> 'hidden'
                WHERE $skus_names
                AND (p.post_parent = 0 or p.post_parent is null)
                AND p.ID not in ($bids)
                AND p.post_status = 'publish'
                GROUP BY p.ID

        "
		);

		$searched_posts         = array();
		$searched_posts         = array_merge( $variable_products, $simple_products );
		$wp_query->found_posts += sizeof( $searched_posts );
		return $searched_posts;
	}


	public function admin_enqueue_scripts( $hook_suffix ) {

		// WooCommerce Orders View
		$post_type           = 'shop_order';
		$hook_suffix_matches = array( 'woocommerce_page_wc-orders' );
		if ( ! WCPP()->is_hpos_enabled() ) {
			$hook_suffix_matches[] = 'edit.php';
			$hook_suffix_matches[] = 'post.php';
		}
		if ( in_array( $hook_suffix, $hook_suffix_matches ) ) {
			$screen = get_current_screen();

			if ( is_object( $screen ) && $post_type == $screen->post_type ) {
				// Register, enqueue scripts and styles here
				wp_enqueue_script(
					'shop_order-woocommerce-pickingpal',
					$this->plugin_assets_url . '/js/shop_order.js',
					array( 'jquery' ),
					WCPP_VERSION
				);
				// Pass WordPress environment vars to script
				wp_localize_script(
					'shop_order-woocommerce-pickingpal',
					'WPEnv',
					array(
						'adminUrl' => admin_url(),
					)
				);
			}
		}
	}
}
