<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}



/**
 * WCPP_admin
 *
 * WooCommerce Picking Pal Admin
 *
 * @class       WCPP_admin
 * @category    Class
 */
class WCPP_admin {

	/**
	 * @var object Active tab class instance
	 */
	private $active_tab_class;

	/**
	 * @var string Active tab class name
	 */
	private $active_tab;

	private $initialized = false;

	/**
	 * @var array Array of tab names on main page
	 */
	public $tab_names = array();

	/**
	 * @const Admin Menu slug
	 */
	const MENU_SLUG = 'PickingPal';

	public function __construct() {
		$this->init();
	}

	public function init_tab() {
		if ( ! isset( $_REQUEST['page'] ) or $_REQUEST['page'] != self::MENU_SLUG ) {
			return null;
		}

		// check the logged in user capability and display appropriate tabs
		// include pluggable.php for the current_user_can function to work.
		if ( ! function_exists( 'wp_get_current_user' ) ) {
			require_once ABSPATH . 'wp-includes/pluggable.php';
		}
		global $current_user_can;
		if ( ! current_user_can( 'create_users' ) ) {
			unset( $this->tab_names['settings'] );
			if ( ! WCPP()->is_lite() ) {
				unset( $this->tab_names['license'] );
			}
		}

		if ( isset( $_REQUEST['tab'] ) ) {
			$active_tab = $_REQUEST['tab'];
		} else {
			$active_tab = 'pickingpal';
		}

		$this->active_tab       = $active_tab;
		$tab_obj                = $this->get_tab_obj( $active_tab );
		$this->active_tab_class = $tab_obj ? $tab_obj : false;
		// do POST
		if ( $this->active_tab_class ) {
			$this->active_tab_class->post_handler();
		}
	}

	private function get_tab_obj( $name ) {

		$pro_extension = WCPP()->get_pro_extension();
		if ( $pro_extension ) {
			if ( $tab = $pro_extension->get_tab( $name ) ) {
				return $tab;
			}
		}

		$standard_extension = WCPP()->get_standard_extension();
		if ( $standard_extension ) {
			if ( $tab = $standard_extension->get_tab( $name ) ) {
				return $tab;
			}
		}

		$path       = self::admin_tab_classes_path() . '/class-wcpp-' . $name . '-tab.php';
		$class_name = 'WCPP_' . ucfirst( $name ) . '_Tab';
		if ( file_exists( $path ) ) {
			include_once $path;
			return new $class_name();
		}

		return null;
	}

	/**
	 * Hook actions and filters
	 */
	private function init() {
		if ( ! $this->initialized ) {
			$this->init_tab_names();
			$this->includes();
			add_action( 'admin_init', array( $this, 'init_tab' ) );
			add_action( 'admin_menu', array( $this, 'add_submenu_to_woocommerce' ) );
			add_action( 'admin_init', array( $this, 'print_gate' ) );
			add_action( 'woojetpack_after_settings_save', array( $this, 'clear_order_numbers_transient' ), 10, 2 );

			// add ajax handlers
			foreach ( $this->tab_names as $tab_name => $tab_name_translated ) {
				add_action( 'wp_ajax_' . 'ajax_script_' . $tab_name, array( $this, 'ajax_gate' ) );
			}

			// apply settings to UI
			$settings = array();
			new WCPP_Admin_Tweak_UI( $settings );
		}
		$this->initialized = true;
	}

	private function init_tab_names() {
		$tab_names          = array(
			'pickingpal' => __( 'PickingPal', 'woocommerce-pickingpal' ),
			'log'        => __( 'Log', 'woocommerce-pickingpal' ),
			'license'    => __( 'Upgrade', 'woocommerce-pickingpal' ),
		);
		$standard_extension = WCPP()->get_standard_extension();
		if ( $standard_extension ) {
			$tab_names = $standard_extension->filter_admin_tab_names( $tab_names );
		}
		$pro_extension = WCPP()->get_pro_extension();
		if ( $pro_extension ) {
			$tab_names = $pro_extension->filter_admin_tab_names( $tab_names );
		}
		// move the licence keyed entry to the end of the array (force last tab)
		$tab_names       = array_merge( array_diff_key( $tab_names, array( 'license' => '' ) ), array( 'license' => $tab_names['license'] ) );
		$this->tab_names = $tab_names;
	}

	/**
	 * Include any classes we need within admin.
	 */
	private function includes() {
		include_once self::admin_tab_classes_path() . '/class-wcpp-tab.php';
		include_once 'class-wcpp-admin-tweak-ui.php';
		include_once 'class-wcpp-print-handler.php';
		include_once 'class-wcpp-admin-helper.php';
		include_once 'class-wcpp-picker-data.php';
		include_once 'class-wcpp-print-template.php';
		include_once 'class-wcpp-print-template-data.php';
	}

	public function ajax_gate() {
		if ( isset( $_REQUEST['method'] ) ) {
			$method = $_REQUEST['method'];
			if ( method_exists( $this->active_tab_class, $method ) ) {
				$_POST = array_map( 'stripslashes_deep', $_POST );
				$this->active_tab_class->$method();
			} else {
				echo esc_html( 'AJAX method does not exist' );
			}
		}
		die();
	}

	public function print_gate() {
		$print_handler = new WCPP_Print_Handler();
		$method        = isset( $_REQUEST['method'] ) ? $_REQUEST['method'] : null;
		if ( $method ) {
			$print_handler->handle( $method );
		}
	}

	/**
	 * Get the admin page views paths
	 *
	 * @return string
	 */
	public static function admin_views_path() {
		return untrailingslashit( WCPP()->plugin_views_path() . '/' );
	}

	/**
	 * Get the admin page js paths
	 *
	 * @return string
	 */
	public static function admin_js_url() {
		return untrailingslashit( WCPP()->plugin_js_url() . '/' );
	}

	/**
	 * Get the admin page tab classes paths
	 *
	 * @return string
	 */
	public static function admin_tab_classes_path() {
		return untrailingslashit( WCPP()->plugin_path() . '/includes/admin/tabs/' );
	}

	/**
	 * Add submenu to WooCommerce menu
	 */
	public function add_submenu_to_woocommerce() {

		add_submenu_page(
			'woocommerce',
			'PickingPal', // title menu
			'PickingPal', // submenu label
			'edit_published_pages',
			self::MENU_SLUG,
			array( $this, 'render_submenu_to_woocommerce' )
		);
	}

	/**
	 * Render Pinking Pal tabs without content
	 */
	public function render_submenu_to_woocommerce() {

		$this->render_admin_view(
			'main',
			array(
				'tab_names'        => $this->tab_names,
				'WCPP_admin'       => $this,
				'ajaxurl'          => WCPP()->ajax_url(),
				'menu_slug'        => self::MENU_SLUG,
				'active_tab_class' => $this->active_tab_class,
			)
		);
	}


	public function render_admin_view( $view_name, $params = array() ) {
		extract( $params );
		$path_to_view = null;
		if ( 'main' == $view_name ) {
			$path_to_view = self::admin_views_path() . '/main.php';
		} elseif ( in_array( $view_name, array_keys( $this->tab_names ) ) ) {
			$this->active_tab_class->register();
			$path_to_view = $this->active_tab_class->return_path_to_view();
		}

		if ( ! $path_to_view ) {
			/*
			* Display a simple 404 error for the restricted tabs
			* if accessed via url
			* @TODO consider adding a $path_to_view
			*/
			echo "<div style='margin-top: 105px; margin-left:50%;'><h4>404 page not found</h4></div>";

			return;
		}

		include_once $path_to_view;
	}

	// for JetPack plugin!
	public function clear_order_numbers_transient( $sections, $current_section ) {
		delete_transient( 'pickingpal_orders_numbers_transient' );
	}
}

new WCPP_admin();
