<?php

/**
 * Builds data payload for use in Picker UI
 */

class WCPP_Picker_Data {

	protected $products      = array();
	protected $order         = null;
	protected $order_extend  = null;
	protected $payload       = null;
	protected $bundle_map    = array();
	protected $pick_arr_meta = array();

	public function __construct( $order_id ) {
		$this->order = WC_Order_Compat::get_order_compat( $order_id );
		if ( empty( $this->order ) ) {
			throw new Exception( 'Could not load order ID ' . $order_id );
		}
		$this->order_extend = new WCPP_Order_Extend( $this->order );
		$this->load_bundle_map();
	}

	public function get_data() {
		if ( ! $this->payload ) {
			$this->load_data();
		}
		$this->update_order();

		return apply_filters( 'wc_pickingpal_get_order_items', $this->payload );
	}

	/**
	 * Fetches the picked data stored in meta key
	 * Adds qty_picked field for data stored by version <= 2.3.0
	 *
	 * @param int Order ID
	 *
	 * @return array|false Array of picked item info
	 *
	 * @deprecated
	 * This will become a protected method.
	 * All data related to picked quantities is provided by the get_data() method.
	 */
	public function get_pick_arr( $order_id ) {
		$order = $this->order;
		if ( ! $order || $order_id !== $this->order->get_id() ) {
			$order = WC_Order_Compat::get_order_compat( $order_id );
		}
		return $order ? $this->get_pick_meta( $order ) : false;
	}

	/**
	 * Returns the picker meta data for a given order
	 *
	 * @param $order WC_Order
	 *
	 * @return array
	 */
	protected function get_pick_meta( $order ) {
		if ( ! empty( $this->pick_arr_meta ) ) {
			return $this->pick_arr_meta;
		}
		if ( WCPP()->is_hpos_enabled() ) {
			$pick_arr = $order->get_meta( WCPP()->get_meta_key( 'picker_data' ), true );
		} else {
			$pick_arr = get_post_meta( $order->get_id(), WCPP()->get_meta_key( 'picker_data' ), true );
		}
		if ( $pick_arr ) {
			foreach ( $pick_arr as $key => $obj ) {
				if ( ! isset( $obj['qty_picked'] ) && isset( $obj['qty'] ) ) {
					$obj['qty_picked'] = $obj['qty'];
				}
				$this->pick_arr_meta[ $key ] = $obj;
			}
		}
		return $this->pick_arr_meta;
	}

	protected function load_data() {
		$orderItems = $this->order->get_items( 'line_item' );
		foreach ( $orderItems as $orderItem ) {
			$this->add_order_item( $orderItem );
		}
		$this->update_products_locations();
		$this->update_products_picked();
		$this->sort_products_by_location();

		$this->order->post          = get_post( $this->order->get_id() );
		$this->order->post_status   = wc_get_order_status_name( $this->order->get_status() );
		$this->order->customer_note = $this->order->get_customer_note();
		$this->order->order_id      = $this->order->get_id();
		$this->order->order_number  = $this->order->get_order_number();
		$is_autopick_digital        = ( WCPP()->settings_handler->get_option( 'autopick_digital_products' ) and 'true' == WCPP()->settings_handler->get_option( 'autopick_digital_products' ) );

		$this->payload = array(
			'order'                       => $this->order,
			'picked'                      => $this->all_items_picked(),
			'products'                    => $this->products,
			'override_sku'                => WCPP()->override_sku,
			'pick_reset_override_sku'     => WCPP()->pick_reset_override_sku,
			'status_switch_barcode_units' => WCPP()->settings_handler->get_option( 'status_switch_via_barcode_units' ) ?: array(),
			'processing'                  => $this->any_items_picked(),
			'autopick_digital'            => $is_autopick_digital,
		);
	}

	protected function add_order_item( $orderItem ) {
		$product_obj = $orderItem->get_product();
		if ( ! $product_obj ) {
			return;
		}

		$product_id        = WCPP_Admin_Helper::is_variation( $product_obj ) ? WCPP_Admin_Helper::get_product_variation_id( $product_obj ) : $product_obj->get_id();
		$product_image     = $product_obj->get_image( array( 38, 38 ) );
		$product_img_src   = wp_get_attachment_image_src(
			get_post_thumbnail_id( $product_obj->get_id() ),
			'single-post-thumbnail'
		);
		$product_image_url = $product_img_src ? $product_img_src[0] : null;
		$additional_skus   = WCPP()->helper->get_product_additional_skus( $product_id, false );

		$formatted_product_name = WCPP_Admin_Helper::get_product_formatted_name( $product_obj );
		$formatted_product_name = $this->strip_name_attributes( $formatted_product_name );

		$orderItemId     = $orderItem->get_id();
		$bundled_by      = isset( $orderItem['bundled_by'] ) ? $orderItem['bundled_by'] : null;
		$bundled_items   = isset( $orderItem['bundled_items'] ) ? $orderItem['bundled_items'] : null;
		$bundle_info     = ( $bundled_by && isset( $this->bundle_map[ $bundled_by ] ) ) ? $this->bundle_map[ $bundled_by ] : array();
		$bundle_id       = isset( $bundle_info['itemId'] ) ? $bundle_info['itemId'] : null;
		$bundle_multiple = isset( $bundle_info['qty_required'] ) ? $bundle_info['qty_required'] : 0;

		$is_parent  = empty( $orderItem['bundled_items'] ) ? 0 : 1;
		$parent     = $bundle_id ? $bundle_id : - 1;
		$item_qty   = $orderItem['qty'];
		$bundle_qty = $bundle_multiple ? max( $item_qty / $bundle_multiple, 1 ) : 0;

		$name = apply_filters(
			'wc_pickingpal_get_title',
			$formatted_product_name,
			$product_id,
			$orderItem,
			$product_obj
		);
		$sku  = apply_filters(
			'wc_pickingpal_get_sku',
			$product_obj->get_sku(),
			$product_id,
			$orderItem,
			$product_obj
		);

		$meta_data     = WCPP()->helper->get_order_item_meta_data_array( $orderItem );
		$qty_restocked = (int) $orderItem->get_meta( '_restock_refunded_items', true );
		$qty_required  = $item_qty;
		if ( $qty_required && $item_qty ) {
			$qty_required = max( 0, $item_qty - $qty_restocked );
		}

		$position              = count( $this->products ) + 1;
		$item_value_before_tax = $orderItem->get_total();
		$item_tax_value        = $orderItem->get_total_tax();

		$variation_description = '';
		$variation_data        = $product_obj->is_type( 'variation' ) ? $product_obj->get_variation_attributes() : array();
		if ( $variation_data ) {
			$variation_description = implode(
				' | ',
				array_map(
					function ( $e ) {
						return $e['label'] . ': ' . $e['value'];
					},
					WCPP_Admin_Helper::get_product_variations( $product_obj )
				)
			);
		}

		$product_data                   = array(
			'sku'                       => $sku,
			'additional_skus'           => $additional_skus,
			'additional_skus_formatted' => implode(
				';',
				array_map(
					function ( $item ) {
						return $item['label'] . ':' . $item['value'];
					},
					$additional_skus
				)
			),
			'name'                      => $name,
			'name_formatted'            => $name,
			'description'               => $variation_description,
			'parent'                    => $parent,
			'isparent'                  => $is_parent,
			'qty'                       => $item_qty,
			'qty_picked'                => 0,
			'qty_required'              => $qty_required,
			'qty_overpicked'            => 0,
			'qty_restocked'             => $qty_restocked,
			'bundle_qty'                => $bundle_qty,
			'backordered'               => null,
			'is_digital'                => WCPP_Admin_Helper::is_digital( $product_obj ),
			'order_item_id'             => $orderItemId,
			'image'                     => $product_image,
			'image_formatted'           => $product_image,
			'image_url'                 => $product_image_url,
			'product_id'                => $product_id,
			'sort_order'                => $position,
			'meta_data'                 => $meta_data,
			'variation_data'            => $product_obj->is_type( 'variation' ) ? $product_obj->get_variation_attributes() : array(),
			'value'                     => $item_value_before_tax,
			'value_formatted'           => wc_price( $item_value_before_tax ),
			'tax'                       => $item_tax_value,
			'warehouses'                => array(),
			'warehouse'                 => '',
		);
		$product_data                   = $this->update_product_backordered( $product_data, $product_obj );
		$this->products[ $orderItemId ] = $product_data;
	}


	protected function load_bundle_map() {
		$orderItems = $this->order->get_items( 'line_item' );
		foreach ( $orderItems as $orderItem ) {
			if ( ! isset( $orderItem['bundled_items'] ) ) {
				continue;
			}
			if ( ! empty( $orderItem['bundled_items'] ) ) {
				$bundle_key                      = $orderItem['bundle_cart_key'];
				$this->bundle_map[ $bundle_key ] = array(
					'itemId'       => $orderItem->get_id(),
					'qty'          => $orderItem['qty'],
					'qty_required' => $orderItem['qty_required'],
				);
			}
		}
	}

	protected function update_product_backordered( $product_data, $product_obj ) {
		$backordered  = null;
		$qty_required = $product_data['qty_required'];

		if ( 'onbackorder' === $product_obj->get_stock_status() ) {
			$backordered = $qty_required;
		} else {
			$order_quant = $qty_required;
			$product_id  = $product_data['product_id'];
			foreach ( $this->products as $entry ) {
				$entry_pid = $entry['product_id'];
				if ( $product_id == $entry_pid ) {
					$order_quant += $entry['qty_required'];
				}
			}

			if ( $product_obj->is_on_backorder( $order_quant ) ) {
				$backordered           = $qty_required;
				$total_stock_available = $product_obj->get_stock_quantity();
				if ( $total_stock_available > 0 ) {
					$stock_remaining = $total_stock_available - $order_quant;
					if ( $stock_remaining < 0 ) {
						$stock_deficit = abs( $stock_remaining );
						$backordered   = min( $qty_required, $stock_deficit );
					}
				}
			}
		}

		if ( $backordered ) {
			$product_data['backordered'] = $backordered;
		}

		return $product_data;
	}

	protected function update_products_locations() {
		//
		// Add warehouse locations to product items
		//
		$warehouse_locations = WCPP()->settings_handler->get_warehouse_locations( 'only_enabled' );
		foreach ( $this->products as $productIndex => $productData ) {
			$productId = isset( $productData['product_id'] ) ? $productData['product_id'] : null;
			if ( $productId ) {
				$item_warehouse_location_labels = array();
				$item_warehouse_locations       = array();
				foreach ( $warehouse_locations as $key => $location ) {
					$_value = get_post_meta( $productId, $key );
					if ( ! empty( $_value[0] ) ) {
						$item_warehouse_locations[ $key ] = array(
							'label' => $location['label'],
							'value' => $_value[0],
						);
					}
				}
				$warehouse_locations_formatted                           = implode(
					';',
					array_map(
						function ( $item ) {
							return $item['label'] . ':' . $item['value'];
						},
						$item_warehouse_locations
					)
				);
				$this->products[ $productIndex ]['warehouses']           = $item_warehouse_locations;
				$this->products[ $productIndex ]['warehouses_formatted'] = $warehouse_locations_formatted;
				// legacy
				$this->products[ $productIndex ]['warehouse'] = $warehouse_locations_formatted;
			}
		}
	}

	protected function update_products_picked() {
		//
		// Compile Pick Counts
		//
		$order_id = $this->order->get_id();
		$pick_arr = $this->get_pick_meta( $this->order );
		if ( is_array( $pick_arr ) ) {
			foreach ( $pick_arr as $arr ) {
				$product_index   = $arr['order_item_id'];
				$picked_quantity = intval( $arr['qty'] );
				if ( $picked_quantity && isset( $this->products[ $product_index ] ) ) {
					$this->products[ $product_index ]['qty_picked'] = $picked_quantity;
					$required_quantity                              = $this->products[ $product_index ]['qty_required'];
					if ( $picked_quantity > $required_quantity ) {
						$this->products[ $product_index ]['qty_overpicked'] = $picked_quantity - $required_quantity;
						$this->products[ $product_index ]['is_ovr']         = true; // deprecated
					}
				}
			}
		}
	}

	protected function sort_products_by_location() {
		//
		// Warehouse location sorting
		//
		$warehouse_locations = WCPP()->settings_handler->get_warehouse_locations( 'only_enabled' );
		if ( $warehouse_locations && wc_string_to_bool( WCPP()->settings_handler->get_option( 'sort_by_warehouse_location' ) ) ) {
			$this->products = WCPP()->helper->sort_array_empty_last( $this->products, 'warehouse' );
			$this->update_order_item_sort_order();
		}
	}

	protected function update_order_item_sort_order() {
		$position = 0;
		foreach ( $this->products as &$product ) {
			++$position;
			$product['sort_order'] = $position;
		}
	}

	protected function strip_name_attributes( $text ) {
		$text = preg_replace( '#,,+#', ',', $text );
		$text = preg_replace( '#(?<=&ndash;\s),#', '', $text );

		return $text;
	}

	protected function all_items_picked() {
		foreach ( $this->products as $product_data ) {
			$q_picked   = (int) $product_data['qty_picked'];
			$q_required = (int) $product_data['qty_required'];
			if ( $q_picked < $q_required ) {
				return false;
			}
		}

		return true;
	}

	protected function any_items_picked() {
		foreach ( $this->products as $product_data ) {
			$q_picked = (int) $product_data['qty_picked'];
			if ( $q_picked > 0 ) {
				return true;
			}
		}

		return false;
	}

	/**
	 * @TODO out of context, find when and where to run this block.
	 */
	protected function update_order() {
		//
		// UPDATE ORDER
		//
		$picked     = $this->all_items_picked();
		$processing = $this->any_items_picked();
		if ( ! $picked && ! $processing ) {
			$this->order_extend->set_pick_status( WCPP_Order_Extend::PICK_STATUS_NOT_SCANNED );
		} elseif ( ! $picked && $processing ) {
			$this->order_extend->set_pick_status( WCPP_Order_Extend::PICK_STATUS_PROCESSING );
		} elseif ( $picked && $processing ) {
			$this->order_extend->set_pick_status( WCPP_Order_Extend::PICK_STATUS_FINISHED );
		}
		$this->order_extend->set_order_number( $this->order->get_order_number() );
		$this->order_extend->save();
	}
}
