<?php
/**
 * This class assists with the rendering of print templates
 * This class is used to handle print requests from URL.
 * The $print_methods array contains a list of methods that can be called from URL
 * to render a print template
 *
 * @package WooCommerce_PickingPal
 */

class WCPP_Print_Handler {


	/**
	 * @var string[] List of methods that can be called from URL to render a print template
	 */
	protected $print_methods = array(
		'pickingpal_print_barcode',
		'pickingpal_print_exact_barcode',
		'pickingpal_print_pick_ticket',
		'pickingpal_print_packing_slip',
	);


	/**
	 * This method handles the print request from URL
	 *
	 * @param string $request_method The name of the method to call
	 *
	 * @return void Called methods will output content to print
	 */
	public function handle( $request_method ) {
		if ( $request_method ) {
			if ( in_array( $request_method, $this->print_methods ) && method_exists( $this, $request_method ) ) {
				$_POST = array_map( 'stripslashes_deep', $_POST );
				$this->$request_method();
			}
		}
	}


	/**
	 * Prints the barcode specified in the request string
	 *
	 * @return void Content wil be output to print
	 * @throws Exception
	 */
	private function pickingpal_print_exact_barcode() {
		$barcode = isset( $_REQUEST['barcode'] ) ? $_REQUEST['barcode'] : '';
		if ( ! $barcode ) {
			exit();
		}
		$this->pickingpal_print_barcode( $barcode );
	}


	/**
	 * Prints the override_sku barcode by default, or other if presented in $barcode param
	 *
	 * @param null|string $barcode Optional barcode to print, otherwise the override sku will be used
	 *
	 * @return void Content wil be output to print
	 * @throws Exception
	 */
	protected function pickingpal_print_barcode( $barcode = null ) {
		$legacy_template_name  = 'barcode';
		$legacy_theme_template = WCPP()->find_theme_template( $legacy_template_name );
		if ( $legacy_theme_template ) {
			$this->print_barcode();
		} else {
			// Render new implementation
			$barcode  = $barcode ? $barcode : WCPP()->override_sku;
			$template = new WCPP_Print_Template( 'template_print_barcode' );
			$template->add_template_data( 'barcode', $barcode );
			$template->render_template();
		}
		exit();
	}

	/**
	 * Prints the pick ticket template
	 *
	 * @return void Content wil be output to print
	 */
	private function pickingpal_print_pick_ticket() {
		$legacy_template_name  = 'template_list';
		$legacy_theme_template = WCPP()->find_theme_template( $legacy_template_name );
		if ( $legacy_theme_template ) {
			$this->print_pick_ticket();
		} else {
			// Render new implementation
			$orders   = self::get_orders();
			$template = new WCPP_Print_Template( 'template_print_pick_ticket', $orders );
			$template->render_template();
			$this->mark_orders_printed( $orders, 'pick_ticket' );
		}
		exit();
	}

	/**
	 * Prints the packing slip template
	 *
	 * @return void Content wil be output to print
	 * @throws Exception
	 */
	private function pickingpal_print_packing_slip() {
		$legacy_template_name  = 'template_slip';
		$legacy_theme_template = WCPP()->find_theme_template( $legacy_template_name );
		if ( $legacy_theme_template ) {
			// Support old custom implementation
			$this->print_slip();
		} else {
			// Render new implementation
			$orders   = self::get_orders();
			$template = new WCPP_Print_Template( 'template_print_packing_slip', $orders );
			$template->render_template();
		}
		exit;
	}


	/**
	 * Loads and returns orders from IDs presented in URL request params
	 *
	 * @return WC_Order_Compat[] Returns array of order data
	 */
	protected static function get_orders() {
		$orders = array();
		if ( isset( $_GET['ids'] ) ) {
			foreach ( $_GET['ids'] as $order_id ) {
				$order = WC_Order_Compat::get_order_compat( $order_id );
				if ( ! $order ) {
					$order_id = @apply_filters( 'woocommerce_shortcode_order_tracking_order_id', $order_id );
					$order    = WC_Order_Compat::get_order_compat( $order_id );
				}
				if ( ! $order ) {
					continue;
				}
				$orders[] = WC_Order_Compat::get_order_compat( $order_id );
			}
		}
		return $orders;
	}

	/**
	 * Uses legacy method to output template
	 *
	 * @param $template_name
	 * @param $params
	 *
	 * @return void Content wil be output to print
	 */
	protected function print_legacy_template( $template_name, $params ) {
		WCPP_Admin_Helper::wcpp_render_template(
			$template_name,
			$params
		);
	}

	private function mark_orders_printed( $orders, $template_id ) {
		$meta_key_id = 'printed_' . $template_id;
		if ( $orders ) {
			foreach ( $orders as $order ) {
				if ( WCPP()->is_hpos_enabled() ) {
					$order->add_meta_data( WCPP()->get_meta_key( $meta_key_id ), 'Y', true );
					$order->save_meta_data();
				} else {
					add_post_meta( $order->get_id(), WCPP()->get_meta_key( $meta_key_id ), 'Y', true );
				}
			}
		}
	}


	/**
	 * @return void
	 * @deprecated
	 * @use pickingpal_print_pick_ticket
	 */
	private function print_pick_ticket() {
		$orders        = self::get_orders();
		$template_name = 'template_list';
		$params        = array(
			'orders'       => $orders,
			'url_list_css' => self::get_url_list_css( $template_name ),
			'url_plugin'   => WCPP()->plugin_url(),
		);
		$this->print_legacy_template( $template_name, $params );
		$this->mark_orders_printed( $orders, 'pick_ticket' );
	}

	/**
	 * @return void
	 * @throws Exception
	 * @deprecated
	 * @use pickingpal_print_packing_slip
	 */
	private function print_slip() {
		$template_name = 'template_slip';
		$orders        = self::get_orders();

		if ( ! isset( $orders[0] ) ) {
			exit();
		}

		// LOAD PICKER DATA
		$order_id    = $orders[0]->get_id();
		$picker_data = new WCPP_Picker_Data( $order_id );
		$order_items = $picker_data->get_data();

		// FILTER PICKER DATA!!
		$v_control           = WCPP_Visibility_Controller::init();
		$with_product_images = $v_control->is_visible(
			$v_control::PRODUCT_IMAGE_FEATURE,
			$v_control::PACKING_SLIP_VIEW
		);
		$with_customer_notes = $v_control->is_visible(
			$v_control::CUSTOMER_ORDER_NOTES_FEATURE,
			$v_control::PACKING_SLIP_VIEW
		);

		$with_warehouse_locations = count(
			array_filter(
				WCPP()->settings_handler->get_warehouse_locations( 'only_enabled' ),
				function ( $data, $key ) use ( $v_control ) {
					return $v_control->is_visible( $key, $v_control::PACKING_SLIP_VIEW );
				},
				ARRAY_FILTER_USE_BOTH
			)
		) > 0;

		$with_additional_skus = count(
			array_filter(
				WCPP()->settings_handler->get_skus( 'only_enabled' ),
				function ( $data, $key ) use ( $v_control ) {
					return $v_control->is_visible( $key, $v_control::PACKING_SLIP_VIEW );
				},
				ARRAY_FILTER_USE_BOTH
			)
		) > 0;

		foreach ( $order_items['products'] as &$product ) {
			// filter warehouses
			$warehouses            = isset( $product['warehouses'] ) && is_array( $product['warehouses'] ) ? $product['warehouses'] : array();
			$product['warehouses'] = array_filter(
				$warehouses,
				function ( $data, $key ) use ( $v_control ) {
					return $v_control->is_visible( $key, $v_control::PACKING_SLIP_VIEW );
				},
				ARRAY_FILTER_USE_BOTH
			);

			$new_warehouse_labels = array();
			foreach ( $product['warehouses'] as $key => $location ) {
				$new_warehouse_labels[] = $v_control->get_feature_label( $key ) . ': ' . $location;
			}
			$product['warehouse'] = implode( '; ', $new_warehouse_labels );

			// filter skus
			$additional_skus = isset( $product['additional_skus'] ) && is_array( $product['additional_skus'] ) ? $product['additional_skus'] : array();

			$product['additional_skus'] = array_filter(
				$additional_skus,
				function ( $data, $key ) use ( $v_control ) {
					return $v_control->is_visible( $key, $v_control::PACKING_SLIP_VIEW );
				},
				ARRAY_FILTER_USE_BOTH
			);

			$new_additional_sku_labels = array();
			foreach ( $product['additional_skus'] as $as_key => $as_data ) {
				$new_additional_sku_labels[] = $as_data['label'] . ': ' . $as_data['value'];
			}
			$product['additional_sku_labels'] = implode( '; ', $new_additional_sku_labels );

			// filter images
			if ( ! $with_product_images ) {
				unset( $product['image'] );
				unset( $product['image_url'] );
			}
		}

		$this->print_legacy_template(
			$template_name,
			array(
				'orders'               => $orders,
				'url_list_css'         => self::get_url_list_css( $template_name ),
				'order_items'          => $order_items,
				'pick_arr'             => $picker_data->get_pick_arr( $order_id ),
				'with_additional_skus' => $with_additional_skus,
				'with_warehouses'      => $with_warehouse_locations,
				'with_product_images'  => $with_product_images,
				'with_customer_notes'  => $with_customer_notes,
			)
		);
	}

	/**
	 * @return void
	 * @deprecated
	 * @use pickingpal_print_barcode
	 */
	private function print_barcode() {
		$template_name = 'barcode';
		$barcode       = WCPP()->override_sku;
		$this->print_legacy_template(
			$template_name,
			array(
				'barcode'      => $barcode,
				'url_list_css' => self::get_url_list_css( $template_name ),
			)
		);
	}

	/**
	 * @return string
	 * @deprecated
	 * Get full url to style file list.css
	 */
	protected static function get_url_list_css( $template_name ) {
		if ( file_exists( get_theme_file_path( 'woocommerce/woocommerce-pickingpal-template/list.css' ) ) ) {
			return get_theme_file_uri( 'woocommerce/woocommerce-pickingpal-template/list.css' );
		}

		return WCPP()->plugin_templates_url() . '/list.css';
	}
}
