<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}
require_once 'wc-am-client.php';

/**
 * WCPP_Frontend
 *
 * PickingPal modifies WooCommerce UI
 *
 * @class       WCPP_Frontend
 * @category    Class
 */
class WCPP_Api_Manager extends WC_AM_Client_2_9_3 {
	const PRO_PRODUCT_ID           = 1513;
	const STANDARD_PRODUCT_ID      = 1360;
	private $license_transient_key = 'wcpp_license_state';
	private $coms_fail_transient_key = 'wcpp_coms_fail';
	private $logger;
	private $logger_context = array( 'source' => 'picking-pal-am-client' );

	private static $instance = null;


	public function __construct(
		$file,
		$product_id,
		$software_version,
		$plugin_or_theme,
		$api_url,
		$software_title = '',
		$text_domain = ''
	) {

		add_action(
			'woocommerce_init',
			function () {
				$this->logger = function_exists( 'wc_get_logger' ) ? wc_get_logger() : false;
			}
		);

		$menu = array(
			'page_title' => WCPP()->get_app_name() . ' License',
			'menu_title' => 'PickingPal License',
		);
		parent::__construct(
			$file,
			$product_id,
			$software_version,
			$plugin_or_theme,
			$api_url,
			$software_title,
			$text_domain,
			$menu
		);

		if ( is_admin() ) {

			add_action(
				'admin_init',
				function ( $product_id ) {
					if ( !$this->get_api_key_status(false) || !$this->is_activated_license() ) {
							add_action( 'admin_notices', array( $this, 'inactive_notice' ) );
					}
				}
			);

			// Clear the cached activation status if woocommerce tool clears transients
			$transient_key = $this->get_license_transient_key();
			add_action(
				'woocommerce_system_status_tool_executed',
				function ( $tool ) use ( $transient_key ) {
					if ( 'clear_transients' == $tool['id'] ) {
						delete_transient( $transient_key );
					}
				},
				10,
				1
			);

			// Clear the cached activation status if the activation option is updated by API manager
			add_action(
				'update_option',
				function ( $option, $old_value, $value ) use ( $transient_key ) {
					if ( strpos( $option, 'wc_am_client_' ) !== false && strpos( $option, '_activated' ) !== false ) {
						if ( $old_value !== $value ) {
							delete_transient( $transient_key );
						}
					}
				},
				10,
				3
			);

			// Clear the cached activation status if the plugin is upgraded
			add_action(
				'upgrader_process_complete',
				function ( $upgrader_object, $options ) use ( $transient_key ) {
					// Check if the updated type is plugins
					if ( $options['action'] === 'update' && $options['type'] === 'plugin' ) {
						foreach ( $options['plugins'] as $plugin ) {
							// Check if the updated plugin is WCPP
							if ( $plugin === plugin_basename( WCPP_PLUGIN_FILE ) ) {
								delete_transient( $transient_key );
							}
						}
					}
				},
				10,
				2
			);
		}
	}

	public static function get() {
		if ( ! self::$instance ) {
			$product_id     = self::get_product_id();
			self::$instance = new self(
				WCPP_PLUGIN_FILE,
				$product_id,
				WCPP()->get_version(),
				'plugin',
				WCPP()->get_website(),
				WCPP()->get_app_name(),
				''
			);
		}
		return self::$instance;
	}

	/**
	 * This method will check to see if the license key is activated.
	 * A transient is used to cache the result for 24 hours, limiting the number of API calls.
	 *
	 * If the transient does not exist, or is expired, a API call will be made to pickingpal.com
	 *
	 * If the API call fails, the last known status will be returned and cached for 24 hours.
	 *
	 * If the API call fails for 7 days, the license will be deactivated.
	 *
	 * @param bool $live Ignore 24 hour cache and make a live request to the server
	 *
	 * @return bool
	 */
	public function is_activated_license( $live = false ) {

		if ( ! is_admin() ) {
			return true; // Always return true if not in the admin area
		}

		$transient_key = $this->get_license_transient_key();
		if ( $live === false ) {
			$cached_status = get_transient( $transient_key );
			if ( $cached_status !== false && is_array( $cached_status ) && isset( $cached_status['status'] ) ) {
				return (bool) $cached_status['status'];
			}
		}

		$license_status = $this->license_key_status(true); //if server request fails this will be empty
		if(empty($license_status) ) {
			$this->log_failed_coms();
			if($this->count_failed_coms() < 7){
				$is_activated = $this->get_api_key_status(false); //last known status
			} else {
				$is_activated = false;
			}
		} else {
			$this->reset_failed_coms();
			$is_activated = isset($license_status['data']['activated']) && $license_status['data']['activated'];
		}
		set_transient( $transient_key, array( 'status' => $is_activated ), SECONDS_IN_DAY );
		return $is_activated;
	}



	/**
	 * Generate the default data.
	 */
	public function activation() {
		// Check if we are on a Multisite or not.
		if ( is_multisite() ) {
			$site_ids = get_sites( array( 'fields' => 'ids' ) );

			// Uninstall the plugin for all these sites.
			foreach ( $site_ids as $site_id ) {
				switch_to_blog( $site_id );
				parent::activation();
				restore_current_blog();
			}
		} else {
			parent::activation();
		}
	}

	private function get_license_transient_key() {
		return $this->license_transient_key . '_' . self::get_product_id();
	}

	/**
	 * This is the value assigned to private variable `wc_am_product_id`
	 * when no product id is passed to WC_AM_Client_x
	 *
	 * @return string
	 */
	private function get_product_id_field_name() {
		$identifier = dirname( untrailingslashit( plugin_basename( WCPP_PLUGIN_FILE ) ) );
		$product_id = strtolower(
			str_ireplace(
				array(
					' ',
					'_',
					'&',
					'?',
					'-',
				),
				'_',
				$identifier
			)
		);
		return 'wc_am_product_id_' . $product_id;
	}

	private function log_failed_coms() {
		$failed_coms = $this->count_failed_coms();
		// Check if the transient already exists
		if ($failed_coms === 0) {
			// If no transient exists, initialize fail count and expiration timestamp
			$transient_data = [
				'fails' => 1,
				'expires' => time() + (SECONDS_IN_DAY * 7) // 7 days from now
			];
		} else {
			$transient_data = get_transient($this->coms_fail_transient_key);
			$transient_data['fails']++;
		}
		set_transient($this->coms_fail_transient_key, $transient_data);
	}

	private function count_failed_coms() {
		$transient_data = get_transient($this->coms_fail_transient_key);
		if($transient_data && isset($transient_data['fails']) && isset($transient_data['expires'])) {
			$remaining_time = $transient_data['expires'] - time();
			if ($remaining_time > 0) {
				return $transient_data['fails'];
			}
		}
		return 0;
	}

	private function reset_failed_coms(){
		delete_transient($this->coms_fail_transient_key);
	}


	public static function get_product_id() {
		return WCPP()->is_pro() ? self::PRO_PRODUCT_ID : self::STANDARD_PRODUCT_ID;
	}
}
