<?php

namespace Ademti\WoocommerceProductFeeds\Cache;

use ActionScheduler_Store;
use Ademti\WoocommerceProductFeeds\Helpers\DebugService;
use Ademti\WoocommerceProductFeeds\Dependencies\Pimple\Container;

/**
 * Class fCache
 *
 * @SuppressWarnings(PHPMD.ExcessivePublicCount)
 */
class Cache {

	// Dependencies.
	protected Container $container;
	protected DebugService $debug_service;

	/**
	 * @var array
	 */
	private static array $jobs = [];

	/**
	 * Whether to use the render cache.
	 *
	 * @var boolean
	 */
	private bool $cache_enabled = false;

	/**
	 * Constructor.
	 *
	 * @param DebugService $debug_service
	 * @param Container $container
	 */
	public function __construct( DebugService $debug_service, Container $container ) {
		$this->debug_service = $debug_service;
		$this->container     = $container;
	}

	/**
	 *  Work out if the cache is enabled or not. Trigger initialisation of worker
	 *  processes.
	 *
	 * @return void
	 */
	public function initialise(): void {
		add_action( 'plugins_loaded', [ $this, 'enable_cache' ] );
		add_action( 'init', [ $this, 'init_workers' ], 9 );
		add_filter( 'woocommerce_gpf_render_cache_enabled', [ $this, 'toggle_cache_in_debug_mode' ], 99 );
	}

	/**
	 * Enable the cache via a filter if required.
	 */
	public function enable_cache(): void {
		// Cache is disabled by default. It can be enabled via a filter.
		$this->cache_enabled = apply_filters( 'woocommerce_gpf_render_cache_enabled', $this->cache_enabled );
	}

	/**
	 * Allows the cache to be forcibly enabled / disabled via a URL arg when in debug mode.
	 */
	public function toggle_cache_in_debug_mode( $cache_active ) {
		// phpcs:disable WordPress.Security.NonceVerification.Recommended
		if ( ! $this->debug_service->debug_active() || ! isset( $_GET['force_cache'] ) ) {
			return $cache_active;
		}
		if ( '1' === $_GET['force_cache'] ) {
			return true;
		}
		if ( '0' === $_GET['force_cache'] ) {
			return false;
		}

		return $cache_active;
		// phpcs:enable WordPress.Security.NonceVerification.Recommended
	}

	/**
	 * Init instances for all cache jobs.
	 *
	 * @return void
	 */
	public function init_workers() {
		// Bail if we've already created instances.
		if ( ! empty( self::$jobs ) || ! $this->cache_enabled ) {
			return;
		}
		// Instantiate worker queues.
		$job_types = [
			'ClearAllJob',
			'ClearProductJob',
			'RebuildSimpleJob',
			'RebuildComplexJob',
			'RebuildProductJob',
		];
		foreach ( $job_types as $job_type ) {
			self::$jobs[ $job_type ] = $this->container[ $job_type ];
		}
	}

	/**
	 * Allow external classes to see if the cache is enabled.
	 *
	 * @return boolean  True if the cache is enabled. False otherwise.
	 */
	public function is_enabled() {
		return $this->cache_enabled;
	}

	/**
	 * Fetch multiple items from the cache.
	 *
	 * @param array $post_ids Array of post IDs
	 * @param string $name The cache name to get for these items.
	 *
	 * @return array                 Array of post_id => cached_value for all matched items.
	 */
	public function fetch_multi( $post_ids, $name ) {
		global $wpdb, $table_prefix;

		if ( ! $this->cache_enabled ) {
			return [];
		}

		$cache_name = apply_filters( 'woocommerce_gpf_cache_name', $name );

		$placeholders = array_fill( 0, count( $post_ids ), '%d' );
		$placeholders = implode( ', ', $placeholders );
		$sql          = "SELECT `post_id`, `value`
		          FROM {$table_prefix}wc_gpf_render_cache
				 WHERE `post_id` IN ($placeholders)
				   AND `name` = %s";
		$post_ids[]   = $cache_name;
		// phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
		$results = $wpdb->get_results( $wpdb->prepare( $sql, $post_ids ), OBJECT_K );
		$results = wp_list_pluck( $results, 'value', 'post_id' );

		return $results;
	}

	/**
	 * Fetch an item from the cache.
	 *
	 * @param int $post_id The post ID that this item is attached to.
	 * @param string $name The cache name to get for this item.
	 *
	 * @return string|null           Cached value, or null.
	 */
	public function fetch( $post_id, $name ) {
		global $wpdb, $table_prefix;

		if ( ! $this->cache_enabled ) {
			return null;
		}
		$cache_name = apply_filters( 'woocommerce_gpf_cache_name', $name );

		return $wpdb->get_var(
			$wpdb->prepare(
				'SELECT `value`
				          FROM %i
						 WHERE `post_id` = %d
						   AND `name` = %s',
				$table_prefix . 'wc_gpf_render_cache',
				$post_id,
				$cache_name
			)
		);
	}

	/**
	 * Store / update an item in the cache.
	 *
	 * @param int $post_id The post ID that this item is attached to.
	 * @param string $name The cache name to get for this item.
	 * @param string $value The value to store.
	 *
	 * @return void
	 */
	public function store( $post_id, $name, $value ) {
		global $wpdb, $table_prefix;
		if ( ! $this->cache_enabled ) {
			return;
		}
		$cache_name = apply_filters( 'woocommerce_gpf_cache_name', $name );
		$cache_id   = $wpdb->get_var(
			$wpdb->prepare(
				'SELECT `id`
				   FROM %i
				  WHERE `post_id` = %d
				    AND `name` = %s',
				$table_prefix . 'wc_gpf_render_cache',
				$post_id,
				$cache_name
			)
		);
		if ( is_null( $cache_id ) ) {
			$wpdb->query(
				$wpdb->prepare(
					'INSERT INTO %i
					             (`post_id`, `name`, `value`)
						  VALUES ( %d, %s, %s )',
					$table_prefix . 'wc_gpf_render_cache',
					$post_id,
					$cache_name,
					$value
				)
			);
		} else {
			$wpdb->query(
				$wpdb->prepare(
					'UPDATE %i
					    SET `value` = %s
					  WHERE id = %d',
					$table_prefix . 'wc_gpf_render_cache',
					$value,
					$cache_id
				)
			);
		}
	}

	/**
	 * Drop a specific product's data from the cache, and request a rebuild for it.
	 *
	 * @param int $post_id The product's post ID to be cleared down.
	 *
	 * @return void
	 */
	public function flush_product( $post_id ) {
		if ( ! $this->cache_enabled ) {
			return;
		}
		$pending = as_get_scheduled_actions(
			[
				'hook'     => 'woocommerce_product_feeds_cache_rebuild_product',
				'args'     => [ $post_id ],
				'status'   => ActionScheduler_Store::STATUS_PENDING,
				'per_page' => 1,
				'orderby'  => 'none',
			],
			'ids'
		);
		if ( empty( $pending ) ) {
			as_schedule_single_action(
				time() + 5, // We delay this to allow any WooCommerce cache update jobs to complete first
				'woocommerce_product_feeds_cache_rebuild_product',
				[
					$post_id,
				],
				'woocommerce-product-feeds'
			);
		}
	}

	/**
	 * @return void
	 */
	public function clear_product( $post_id ) {
		if ( ! $this->cache_enabled ) {
			return;
		}
		$pending = as_get_scheduled_actions(
			[
				'hook'     => 'woocommerce_product_feeds_cache_clear_product',
				'args'     => [ $post_id ],
				'status'   => ActionScheduler_Store::STATUS_PENDING,
				'per_page' => 1,
				'orderby'  => 'none',
			],
			'ids'
		);
		if ( empty( $pending ) ) {
			as_schedule_single_action(
				time(),
				'woocommerce_product_feeds_cache_clear_product',
				[
					$post_id,
				],
				'woocommerce-product-feeds'
			);
		}
	}

	/**
	 * Drop objects from the cache, and request a rebuild for them.
	 *
	 * We queue a RebuildProductJob. That will validate that the object is
	 * indeed a product before acting, and ignore it if not.
	 *
	 * @param array $post_ids The object IDs to be cleared down.
	 *
	 * @return void
	 */
	public function flush_objects( $post_ids ) {
		if ( ! $this->cache_enabled ) {
			return;
		}
		foreach ( $post_ids as $post_id ) {
			as_schedule_single_action(
				time(),
				'woocommerce_product_feeds_cache_rebuild_product',
				[
					$post_id,
				],
				'woocommerce-product-feeds'
			);
		}
	}

	/**
	 * Flush any products with a specific term, and rebuild them.
	 *
	 * @SuppressWarnings(PHPMD.UnusedFormalParameter)
	 *
	 * @return void
	 */
	public function flush_term( int $term_id, $tt_id, $taxonomy ) {
		if ( ! $this->cache_enabled ) {
			return;
		}
		as_schedule_single_action(
			time(),
			'woocommerce_product_feeds_cache_rebuild_simple',
			[
				0,
				apply_filters( 'woocommerce_product_feeds_rebuild_chunk_limit_simple', 30 ),
				[
					'taxonomy' => $taxonomy,
					'term_id'  => $term_id,
				],
			],
			'woocommerce-product-feeds'
		);
		as_schedule_single_action(
			time(),
			'woocommerce_product_feeds_cache_rebuild_complex',
			[
				0,
				apply_filters( 'woocommerce_product_feeds_rebuild_chunk_limit_complex', 1 ),
				[
					'taxonomy' => $taxonomy,
					'term_id'  => $term_id,
				],
			],
			'woocommerce-product-feeds'
		);
	}

	/**
	 * Clear the cache, and trigger a rebuild.
	 *
	 * @SuppressWarnings(PHPMD.UndefinedVariable)
	 *
	 * @return void
	 */
	public function flush_all() {
		if ( ! $this->cache_enabled || empty( self::$jobs ) ) {
			return;
		}
		foreach ( self::$jobs as $job ) {
			$job->cancel_all();
		}
		as_schedule_single_action(
			time(),
			'woocommerce_product_feeds_cache_clear_all',
			[],
			'woocommerce-product-feeds'
		);
	}
}
