<?php
/**
 * Exclusion list table class
 *
 * @package YITH\AddedToCartPopup\Admin\AdminTables
 */

defined( 'YITH_WACP' ) || exit; // Exit if accessed directly.

if ( ! class_exists( 'WP_List_Table' ) ) {
	require_once ABSPATH . 'wp-admin/includes/class-wp-list-table.php';
}

if ( ! class_exists( 'YITH_WACP_Exclusions_List_Table' ) ) {

	/**
	 * Exclusion list table class.
	 *
	 * @class   YITH_WACP_Exclusions_List_Table
	 * @since   2.0.0
	 * @author  YITH <plugins@yithemes.com>
	 * @package YITH\AddedToCartPopup\Admin\AdminTables
	 */
	class YITH_WACP_Exclusions_List_Table extends WP_List_Table {

		/**
		 * Class constructor method.
		 *
		 * @access public
		 * @since  2.0.0
		 */
		public function __construct() {
			// Set parent defaults.
			parent::__construct(
				array(
					'singular' => 'exclusion',  // singular name of the listed records.
					'plural'   => 'exclusions', // plural name of the listed records.
					'ajax'     => false,        // does this table support ajax?.
				)
			);

			$this->handle_bulk_action();

		}

		/**
		 * Get bulk actions
		 *
		 * @return array
		 * @since  2.0.0
		 */
		public function get_bulk_actions() {
			return array( 'remove' => esc_html__( 'Remove', 'yith-woocommerce-added-to-cart-popup' ) );
		}

		/**
		 * Extra controls to be displayed between bulk actions and pagination, which
		 * includes our Filters: Customers, Products, Availability Dates
		 *
		 * @param string $which the placement, one of 'top' or 'bottom'.
		 *
		 * @since 2.0.0
		 * @see   WP_List_Table::extra_tablenav();
		 */
		public function extra_tablenav( $which ) {
			if ( 'top' === $which ) {
				// Customers, products.
				$this->render_type_filter();

				submit_button(
					__( 'Filter', 'yith-woocommerce-added-to-cart-popup' ),
					'button',
					false,
					false,
					array(
						'id'    => 'post-query-submit',
						'class' => 'yith_wacp_filter_button',
					)
				);
			}
		}

		/**
		 * Show the filter for type.
		 *
		 * @return void
		 * @since  2.0.0
		 */
		protected function render_type_filter() {

			$current_type = isset( $_REQUEST['type'] ) && ! empty( $_REQUEST['type'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['type'] ) ) : '';  // phpcs:ignore WordPress.Security.NonceVerification.Recommended
			$options      = array(
				''            => esc_html__( 'All types', 'yith-woocommerce-added-to-cart-popup' ),
				'product'     => esc_html__( 'Products', 'yith-woocommerce-added-to-cart-popup' ),
				'product_cat' => esc_html__( 'Categories', 'yith-woocommerce-added-to-cart-popup' ),
			);
			?>
			<div class="alignleft actions">
				<select name="type" id="type">
					<?php foreach ( $options as $key => $option ) : ?>
						<option value="<?php echo esc_attr( $key ); ?>" <?php selected( $current_type, $key ); ?>><?php echo esc_html( $option ); ?></option>
					<?php endforeach; ?>
				</select>
			</div>
			<?php
		}

		/**
		 * Print default column content
		 *
		 * @param mixed  $item        Item of the row.
		 * @param string $column_name Column name.
		 *
		 * @return string
		 * @since  2.0.0
		 */
		public function column_default( $item, $column_name ) {
			$column = '';
			switch ( $column_name ) {
				case 'type':
					$column = sprintf( '<span class="yith-wacp-data-type" data-type="%s">%s</span>', $item['type'], $this->get_type_label( $item['type'] ) );
					break;
				case 'name':
					$column = ( 'product' === $item['type'] ? '<div class="name-wrapper">' . $item['image'] . ' ' . $item['name'] . '</div>' : $item['name'] );
					break;
				case 'action':
					$actions['delete'] = array(
						'type'   => 'action-button',
						'title'  => esc_html_x( 'Delete', 'Tip to delete the product inside the exclusion list', 'yith-woocommerce-added-to-cart-popup' ),
						'action' => 'delete',
						'icon'   => 'trash',
						'url'    => '',
						'class'  => 'action__trash',
					);

					$column = yith_plugin_fw_get_action_buttons( $actions );
					break;
			}

			return $column;
		}

		/**
		 * Returns columns available in table
		 *
		 * @return array Array of columns of the table.
		 * @since  2.0.0
		 */
		public function get_columns() {
			$columns = array(
				'cb'     => '<input type="checkbox" />',
				'name'   => esc_html__( 'Name', 'yith-woocommerce-added-to-cart-popup' ),
				'type'   => esc_html__( 'Type', 'yith-woocommerce-added-to-cart-popup' ),
				'action' => '',
			);

			return $columns;
		}

		/**
		 * Column cb.
		 *
		 * @param array $item Instance.
		 *
		 * @return string
		 * @since  2.0.0
		 */
		public function column_cb( $item ) {
			return sprintf( '<input type="checkbox" name="%1$s[%3$s-%2$s]" value="%2$s" />', $this->_args['singular'], $item['id'], $item['type'] );
		}

		/**
		 * Process Bulk Actions*
		 *
		 * @return void
		 * @since  2.0.0
		 */
		public function handle_bulk_action() {

			$action = $this->current_action();
			if ( ! isset( $_REQUEST['_wpnonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_REQUEST['_wpnonce'] ) ), 'bulk-' . $this->_args['plural'] ) ) {
				return;
			}

			$items = isset( $_REQUEST[ $this->_args['singular'] ] ) ? wp_unslash( $_REQUEST[ $this->_args['singular'] ] ) : array(); //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

			if ( $action && ! empty( $items ) ) {

				$exclusions_prod = array_filter( explode( ',', get_option( 'yith-wacp-exclusions-prod-list' ) ) );
				$exclusions_cat  = array_filter( explode( ',', get_option( 'yith-wacp-exclusions-cat-list' ) ) );

				if ( 'remove' === $action ) {
					$update = array();
					foreach ( $items as $key => $item ) {
						$type = explode( '-', $key );

						if ( ! in_array( $type[0], $update, true ) ) {
							$update[] = $type[0];
						}

						switch ( $type[0] ) {
							case 'product':
								$key = array_search( $item, $exclusions_prod, true );
								if ( false !== $key ) {
									unset( $exclusions_prod[ $key ] );
								}
								break;
							case 'product_cat':
								$key = array_search( $item, $exclusions_cat, true );
								if ( false !== $key ) {
									unset( $exclusions_cat[ $key ] );
								}
								break;
						}
					}

					if ( in_array( 'product', $update, true ) ) {
						update_option( 'yith-wacp-exclusions-prod-list', implode( ',', $exclusions_prod ) );
					}

					if ( in_array( 'product_cat', $update, true ) ) {
						update_option( 'yith-wacp-exclusions-cat-list', implode( ',', $exclusions_cat ) );
					}

					isset( $_REQUEST['_wp_http_referer'] ) && wp_safe_redirect( sanitize_text_field( wp_unslash( $_REQUEST['_wp_http_referer'] ) ) );
				}
			}

		}

		/**
		 * Prepare items for table
		 *
		 * @return void
		 * @since  2.0.0
		 */
		public function prepare_items() {

			// sets pagination arguments.
			$per_page     = 20;
			$current_page = $this->get_pagenum();

			// sets columns headers.
			$columns               = $this->get_columns();
			$hidden                = array();
			$sortable              = $this->get_sortable_columns();
			$this->_column_headers = array( $columns, $hidden, $sortable );
			$exclusion_items       = array();
			$current_type          = isset( $_REQUEST['type'] ) && ! empty( $_REQUEST['type'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['type'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Recommended

			if ( empty( $current_type ) || 'product' === $current_type ) {
				$exclusions_products = array_filter( explode( ',', get_option( 'yith-wacp-exclusions-prod-list' ) ) );
				if ( ! empty( $exclusions_products ) ) {
					foreach ( $exclusions_products as $product_id ) {
						$product = wc_get_product( $product_id );
						if ( $product ) {
							$new_item = array(
								'type'  => 'product',
								'id'    => $product_id,
								'name'  => $product->get_formatted_name(),
								'image' => $product->get_image( 'shop_thumbnail' ),
							);

							$exclusion_items[] = $new_item;
						}
					}
				}
			}

			if ( empty( $current_type ) || 'product_cat' === $current_type ) {
				$exclusions_cat = array_filter( explode( ',', get_option( 'yith-wacp-exclusions-cat-list' ) ) );
				if ( ! empty( $exclusions_cat ) ) {
					foreach ( $exclusions_cat as $cat_id ) {
						$category = get_term_by( 'id', $cat_id, 'product_cat' );
						if ( $category ) {
							$new_item = array(
								'type' => 'product_cat',
								'id'   => $cat_id,
								'name' => $category->name,
							);

							$exclusion_items[] = $new_item;
						}
					}
				}
			}

			$total_items = count( $exclusion_items );

			// retrieve data for table.
			$this->items = array_slice( $exclusion_items, ( $current_page - 1 ) * $per_page, $per_page );

			// sets pagination args.
			$this->set_pagination_args(
				array(
					'total_items' => $total_items,
					'per_page'    => $per_page,
					'total_pages' => ceil( $total_items / $per_page ),
				)
			);
		}

		/**
		 * Return the label of the type
		 *
		 * @param string $key type of item.
		 *
		 * @return string
		 * @since  2.0.0
		 */
		private function get_type_label( $key ) {
			$type = array(
				'product'     => esc_html__( 'Product', 'yith-woocommerce-added-to-cart-popup' ),
				'product_cat' => esc_html__( 'Category', 'yith-woocommerce-added-to-cart-popup' ),
			);

			return isset( $type[ $key ] ) ? $type[ $key ] : $key;
		}

	}
}
