<?php
/**
 * Exclusion Handler class
 *
 * @package YITH\AddedToCartPopup\Admin
 */

defined( 'YITH_WACP' ) || exit; // Exit if accessed directly.

if ( ! class_exists( 'YITH_WACP_Admin_Exclusions_Handler' ) ) {

	/**
	 * Implements the YITH_WACP_Admin_Exclusions_Handler class.
	 *
	 * @class    YITH_WACP_Admin_Exclusions_Handler
	 * @since    2.0.0
	 * @author   YITH <plugins@yithemes.com>
	 * @package  YITH\AddedToCartPopup\Admin
	 */
	class YITH_WACP_Admin_Exclusions_Handler {

		use YITH_WACP_Trait_Singleton;

		/**
		 * Constructor method
		 *
		 * @since 2.0.0
		 */
		public function __construct() {

			// save exclusions list.
			add_action( 'admin_init', array( $this, 'handle_save_exclusion' ) );

			// remove item from exclusions list.
			add_action( 'wp_ajax_yith_wacp_delete_from_exclusion_list', array( $this, 'handle_delete_from_exclusion_list' ) );
			add_action( 'wp_ajax_nopriv_yith_wacp_delete_from_exclusion_list', array( $this, 'handle_delete_from_exclusion_list' ) );

		}

		/**
		 * Save the exclusion.
		 *
		 * @return void
		 * @since  2.0.0
		 */
		public function handle_save_exclusion() {

			if ( ! isset( $_POST['_nonce'], $_POST['yith-wacp-exclusion-type'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['_nonce'] ) ), 'yith_wacp_add_exclusions' ) ) {
				return;
			}

			switch ( $_POST['yith-wacp-exclusion-type'] ) {
				case 'add_products':
					if ( isset( $_POST['add_products'] ) ) {
						$this->save_exclusions( wp_unslash( $_POST['add_products'] ), 'prod' ); //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
					}
					break;
				case 'add_categories':
					if ( isset( $_POST['add_categories'] ) ) {
						$this->save_exclusions( wp_unslash( $_POST['add_categories'] ), 'cat' ); //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
					}
					break;
			}

		}

		/**
		 * Handle delete action from exclusion list
		 *
		 * @return void
		 * @since  2.0.0
		 */
		public function handle_delete_from_exclusion_list() {

			if ( ! isset( $_POST['nonce'], $_POST['type'], $_POST['id'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), 'yith_wacp_delete_exclusions' ) ) {
				return;
			}

			switch ( $_POST['type'] ) {
				case 'product':
					$this->delete_exclusion( sanitize_text_field( wp_unslash( $_POST['id'] ) ), 'prod' );
					break;
				case 'product_cat':
					$this->delete_exclusion( sanitize_text_field( wp_unslash( $_POST['id'] ) ), 'cat' );
					break;
			}
			wp_send_json(
				array(
					'success' => 1,
				)
			);

		}

		/**
		 * Save products or categories exclusions
		 *
		 * @param array  $new_items Items to add.
		 * @param string $type      The type of item to add in the exclusion list.
		 *
		 * @return void
		 * @since  2.0.0
		 */
		public function save_exclusions( $new_items, $type ) {

			// Get old items.
			$old_items = array_filter( explode( ',', get_option( "yith-wacp-exclusions-$type-list" ) ) );

			if ( ! is_array( $new_items ) ) {
				$new_items = explode( ',', $new_items );
			}

			// Merge old with new.
			$exclusions = array_merge( $old_items, $new_items );
			$exclusions = implode( ',', $exclusions );
			update_option( "yith-wacp-exclusions-$type-list", $exclusions, 'no' );
		}

		/**
		 * Delete product or categories from exclusions list
		 *
		 * @param string $id   The ID of the product/category.
		 * @param string $type The type of item to remove from the exclusion list.
		 *
		 * @return void
		 * @since  2.0.0
		 */
		public function delete_exclusion( $id, $type = 'prod' ) {

			$exclusions = array_filter( explode( ',', get_option( "yith-wacp-exclusions-$type-list" ) ) );
			$key        = array_search( $id, $exclusions, true );
			if ( false !== $key ) {
				unset( $exclusions[ $key ] );
			}

			update_option( "yith-wacp-exclusions-$type-list", implode( ',', $exclusions ), 'no' );
		}

	}
}
