<?php
/**
 * Frontend Premium class
 *
 * @package YITH\AddedToCartPopup\Frontend
 */

defined( 'YITH_WACP' ) || exit; // Exit if accessed directly.

if ( ! class_exists( 'YITH_WACP_Frontend_Premium' ) ) {
	/**
	 * Frontend class.
	 * The class manage all the frontend behaviors.
	 *
	 * @class   YITH_WACP_Frontend_Premium
	 * @since   1.0.0
	 * @author  YITH <plugins@yithemes.com>
	 * @package YITH\AddedToCartPopup\Frontend
	 */
	class YITH_WACP_Frontend_Premium extends YITH_WACP_Frontend_Extended {

		use YITH_WACP_Trait_Singleton;

		/**
		 * Plugin enable on single product page
		 *
		 * @since 1.1.0
		 * @var boolean
		 */
		public $enable_single = false;

		/**
		 * Plugin enable on archive
		 *
		 * @since 1.1.0
		 * @var boolean
		 */
		public $enable_loop = false;

		/**
		 * Remove action
		 *
		 * @since 1.0.0
		 * @var string
		 */
		public $action_remove = 'yith_wacp_remove_item_cart';

		/**
		 * Add to cart action
		 *
		 * @since 1.0.0
		 * @var string
		 */
		public $action_add = 'yith_wacp_add_item_cart';

		/**
		 * Add to cart action for YITH WooCommerce Frequently Bought Together
		 *
		 * @since 1.4.0
		 * @var string
		 */
		public $action_add_wfbt = 'yith_wacp_add_wfbt_cart';

		/**
		 * Update cart popup action
		 *
		 * @since 1.3.0
		 * @var string
		 */
		public $action_update = 'yith_wacp_update_item_cart';

		/**
		 * Last cart item key added
		 *
		 * @since 1.4.7
		 * @var string
		 */
		public $last_cart_item = '';

		/**
		 * Constructor
		 *
		 * @access public
		 * @since  1.0.0
		 */
		public function __construct() {

			parent::__construct();

			$this->is_mini_cart_active() && YITH_WACP_Mini_Cart::get_instance();

			$this->enable_single = get_option( 'yith-wacp-enable-on-single', yith_wacp_get_default( 'enable-on-single' ) ) === 'yes';
			$this->enable_loop   = get_option( 'yith-wacp-enable-on-archive', yith_wacp_get_default( 'enable-on-archive' ) ) === 'yes';

			add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_premium' ), 20 );

			add_filter( 'woocommerce_add_to_cart_fragments', array( $this, 'add_to_cart_success_ajax' ), 99, 1 );

			add_action( 'wc_ajax_' . $this->action_remove, array( $this, 'remove_item_cart_ajax' ) );
			add_action( 'wc_ajax_' . $this->action_update, array( $this, 'update_item_cart_ajax' ) );
			// No priv AJAX actions.
			add_action( 'wp_ajax_nopriv' . $this->action_remove, array( $this, 'remove_item_cart_ajax' ) );
			add_action( 'wp_ajax_nopriv' . $this->action_update, array( $this, 'update_item_cart_ajax' ) );

			// Add product from single product page Ajax.
			add_action( 'wp_loaded', array( $this, 'add_item_cart_ajax' ), 30 );
			// Add YITH WooCommerce Frequently Bought Products.
			add_action( 'yith_wfbt_group_added_to_cart', array( $this, 'add_wfbt_cart_ajax' ), 100, 3 );
			// Prevent redirect after ajax add to cart.
			add_filter( 'woocommerce_add_to_cart_redirect', array( $this, 'prevent_redirect_url' ), 100, 1 );
			// Prevent WooCommerce option Redirect to the cart page after successful addition.
			add_filter( 'pre_option_woocommerce_cart_redirect_after_add', array( $this, 'prevent_cart_redirect' ), 10, 2 );
			// Prevent add to cart AJAX.
			add_action( 'woocommerce_before_add_to_cart_button', array( $this, 'exclude_single' ) );
			// Add popup message.
			add_action( 'yith_wacp_before_popup_content', array( $this, 'add_message' ), 10, 1 );
			// Add action button to popup.
			add_action( 'yith_wacp_after_popup_content', array( $this, 'add_actions_button' ), 10, 1 );
			// Add related to popup.
			add_action( 'yith_wacp_after_popup_content', array( $this, 'add_related' ), 20, 1 );
			// Add cart info.
			add_action( 'yith_wacp_add_cart_info', array( $this, 'add_cart_info' ), 10 );
			// Add args to popup template.
			add_filter( 'yith_wacp_popup_template_args', array( $this, 'popup_args' ), 10, 1 );
			// Store last cart item key.
			add_action( 'woocommerce_add_to_cart', array( $this, 'store_cart_item_key' ), 10, 6 );
			// Let's filter item data for cart.
			add_filter( 'woocommerce_product_variation_get_name', array( $this, 'filter_get_name_product' ), 1, 2 );
			// Add param to related query args.
			add_filter( 'yith_wacp_related_products_args', array( $this, 'add_related_query_args' ), 10, 1 );

			// YITH Dynamic Pricing & Discount compatibility.
			add_filter( 'ywdpd_skip_cart_process_discounts', array( $this, 'ywdpd_stop_process_discount' ), 1 );

			// YITH Multi Currency Switcher compatibility.
			add_filter( 'yith_wcmcs_is_ajax_request', array( $this, 'set_multi_currency_ajax_request' ) );
		}

		/**
		 * Check if mini cart feature is active
		 *
		 * @return boolean
		 * @since  1.4.0
		 */
		public function is_mini_cart_active() {
			$is_mobile = wp_is_mobile();

			return ( ! $is_mobile && get_option( 'yith-wacp-mini-cart-enable', yith_wacp_get_default( 'mini-cart-enable' ) ) === 'yes' ) || ( $is_mobile && get_option( 'yith-wacp-mini-cart-enable-mobile', yith_wacp_get_default( 'mini-cart-enable-mobile' ) ) === 'yes' );
		}

		/**
		 * Enqueue scripts and styles premium
		 *
		 * @access public
		 * @since  1.0.0
		 */
		public function enqueue_premium() {

			/**
			 * APPLY_FILTERS: yith_wacp_stylesheet_paths
			 *
			 * Handles the stylesheet paths.
			 *
			 * @param array $paths The stylesheet paths.
			 */
			$paths      = apply_filters( 'yith_wacp_stylesheet_paths', array( WC()->template_path() . 'yith-wacp-frontend.css', 'yith-wacp-frontend.css' ) );
			$located    = locate_template( $paths, false, false );
			$search     = array( get_stylesheet_directory(), get_template_directory() );
			$replace    = array( get_stylesheet_directory_uri(), get_template_directory_uri() );
			$stylesheet = ! empty( $located ) ? str_replace( $search, $replace, $located ) : yit_load_css_file( YITH_WACP_ASSETS_URL . '/css/wacp-frontend.css' );

			wp_enqueue_style( 'yith-wacp-frontend', $stylesheet, array(), YITH_WACP_VERSION );
			wp_enqueue_script( 'yith-wacp-frontend', yit_load_js_file( YITH_WACP_ASSETS_URL . '/js/wacp-frontend.js' ), array( 'jquery' ), YITH_WACP_VERSION, true );

			$inline_css = yith_wacp_get_style_options();
			wp_add_inline_style( 'yith-wacp-frontend', $inline_css );

			// Scroll plugin.
			wp_enqueue_style( 'wacp-scroller-plugin-css', yit_load_css_file( YITH_WACP_ASSETS_URL . '/css/perfect-scrollbar/perfect-scrollbar.css' ), array(), YITH_WACP_VERSION );
			wp_enqueue_script( 'wacp-scroller-plugin-js', yit_load_js_file( YITH_WACP_ASSETS_URL . '/js/perfect-scrollbar/perfect-scrollbar.js' ), array( 'jquery' ), YITH_WACP_VERSION, true );

            // Manually enqueue wc-cart-fragments
            wp_enqueue_script( 'wc-cart-fragments' );

			$selectors = array(
				'body.single-product form.cart:not(.in_loop)',
				'body.single-product form.bundle_form',
				'body.singular-product form.cart',
				'.yith-quick-view.yith-inline form.cart',
				'#jckqv form.cart',
			);

			$localized = array(
				'ajaxurl'                             => WC_AJAX::get_endpoint( '%%endpoint%%' ),
				'actionAdd'                           => $this->action_add,
				'actionRemove'                        => $this->action_remove,
				'actionUpdate'                        => $this->action_update,
				/**
				 * APPLY_FILTERS: yith_wacp_loader_image_url
				 *
				 * Handles the loader image.
				 *
				 * @param string $path The loader image path.
				 */
				'loader'                              => apply_filters( 'yith_wacp_loader_image_url', YITH_WACP_ASSETS_URL . '/images/loader.gif' ),
				'enable_single'                       => $this->enable_single,
				'is_mobile'                           => wp_is_mobile(),
				'popup_size'                          => get_option( 'yith-wacp-box-size', yith_wacp_get_default( 'box-size' ) ),
				/**
				 * APPLY_FILTERS: yith_wacp_form_selectors_filter
				 *
				 * Handles the form selectors.
				 *
				 * @param string $selectors The form selectors.
				 */
				'form_selectors'                      => apply_filters( 'yith_wacp_form_selectors_filter', implode( ',', $selectors ) ),
				/**
				 * APPLY_FILTERS: yith_wacp_allow_automatic_popup
				 *
				 * Handles automatic popup.
				 *
				 * @param boolean $value Is automatic popup allowed?
				 */
				'allow_automatic_popup'               => apply_filters( 'yith_wacp_allow_automatic_popup', true ),
				/**
				 * APPLY_FILTERS: yith_wacp_open_popup_selectors
				 *
				 * Handles the open popup selectors.
				 *
				 * @param string $selector The open popup selector.
				 */
				'open_popup_selectors'                => apply_filters( 'yith_wacp_open_popup_selectors', '#yith-wacp-mini-cart' ),
				/**
				 * APPLY_FILTERS: yith_wacp_enable_touchstart_on_suggested_products
				 *
				 * Handles touchstart on suggested product.
				 *
				 * @param boolean $value Is touchstart enabled?
				 */
				'touchstart_on_suggested_products'    => apply_filters( 'yith_wacp_enable_touchstart_on_suggested_products', true ),
				/**
				 * APPLY_FILTERS: yith_wacp_enable_touchstart_on_popup_actions_buttons
				 *
				 * Handles touchstart on action buttons.
				 *
				 * @param boolean $value Is touchstart enabled?
				 */
				'touchstart_on_popup_actions_buttons' => apply_filters( 'yith_wacp_enable_touchstart_on_popup_actions_buttons', true ),
				'current_currency'                    => ( yith_wacp_is_ywcmcs_enabled() && ! empty( WC()->session ) ) ? WC()->session->get( 'yith_wcmcs_client_currency_id' ) : '',
				'scroll_top_popup_content'            => true
			);

			if ( get_option( 'yith-wacp-enable-wfbt', yith_wacp_get_default( 'enable-wfbt' ) ) === 'yes' ) {
				$localized['actionAddFBT'] = $this->action_add_wfbt;
				$localized['nonceFBT']     = wp_create_nonce( 'yith_bought_together' );
			}

			/**
			 * APPLY_FILTERS: yith_wacp_frontend_script_localized_args
			 *
			 * Handles script localization arguments.
			 *
			 * @param array $localized Localization arguments.
			 */
			wp_localize_script( 'yith-wacp-frontend', 'yith_wacp', apply_filters( 'yith_wacp_frontend_script_localized_args', $localized ) );
		}

		/**
		 * Add args to popup template
		 *
		 * @param mixed $args An array of popup arguments.
		 *
		 * @return mixed
		 * @since  1.0.0
		 */
		public function popup_args( $args ) {
			// Set new animation.
			$args['animation'] = get_option( 'yith-wacp-box-animation', yith_wacp_get_default( 'box-animation' ) );

			return $args;
		}

		/**
		 * Get content html for added to cart popup
		 *
		 * @access public
		 *
		 * @param boolean|object $product  Current product added.
		 * @param string         $layout   Layout to load.
		 * @param string|integer $quantity Product quantity added.
		 *
		 * @return string
		 * @since  1.0.0
		 */
		public function get_popup_content( $product, $layout = '', $quantity = 1 ) {

			if ( ! $layout ) {
				$layout = get_option( 'yith-wacp-layout-popup', yith_wacp_get_default( 'layout-popup' ) );
			}

			// Switch to layout cart if product for some reason is not set or is an YITH Composite.
			if ( 'product' === $layout && ( ! ( $product instanceof WC_Product ) || $product->is_type( 'yith-composite' ) ) ) {
				$layout = 'cart';
			}

			/**
			 * APPLY_FILTERS: yith_wacp_get_popup_content
			 *
			 * Handles popup content arguments.
			 *
			 * @param array $args Popup content arguments.
			 */
			$args = apply_filters(
				'yith_wacp_get_popup_content',
				array(
					'thumb'              => get_option( 'yith-wacp-show-thumbnail', yith_wacp_get_default( 'show-thumbnail' ) ) === 'yes',
					'product_info'       => get_option( 'yith-wacp-show-info', yith_wacp_get_default( 'show-info' ) ) === 'yes',
					'cart_total'         => get_option( 'yith-wacp-show-cart-totals', yith_wacp_get_default( 'show-cart-totals' ) ) === 'yes',
					'cart_shipping'      => get_option( 'yith-wacp-show-cart-shipping', yith_wacp_get_default( 'show-cart-shipping' ) ) === 'yes',
					'cart_tax'           => get_option( 'yith-wacp-show-cart-tax', yith_wacp_get_default( 'show-cart-tax' ) ) === 'yes',
					'product'            => $product,
					'quantity'           => $quantity,
					'last_cart_item_key' => $this->last_cart_item,
				)
			);

			// If $layout is cart just define constant WOOCOMMERCE_CART to get full compatibility.
			( 'cart' === $layout && ! defined( 'WOOCOMMERCE_CART' ) ) && define( 'WOOCOMMERCE_CART', true );

			// Force dynamic to recalculate discount if needed.
			remove_filter( 'ywdpd_skip_cart_process_discounts', array( $this, 'ywdpd_stop_process_discount' ), 1 );

			ob_start();

			/**
			 * DO_ACTION: yith_wacp_before_popup_content
			 *
			 * Perform actions before showing the popup.
			 *
			 * @param WC_Product $product The added product.
			 */
			do_action( 'yith_wacp_before_popup_content', $product );

			wc_get_template( 'yith-wacp-popup-' . $layout . '.php', $args, '', YITH_WACP_TEMPLATE_PATH . '/' );

			/**
			 * DO_ACTION: yith_wacp_after_popup_content
			 *
			 * Perform actions after showing the popup.
			 *
			 * @param WC_Product $product The added product.
			 */
			do_action( 'yith_wacp_after_popup_content', $product );

			return ob_get_clean();
		}

		/**
		 * Added to cart success popup box
		 *
		 * @param array $datas An array of popup data.
		 *
		 * @return array
		 * @since  1.0.0
		 */
		public function add_to_cart_success_ajax( $datas ) {

			// phpcs:disable WordPress.Security.NonceVerification.Recommended
			if ( ! isset( $_REQUEST['product_id'] ) || ( ! isset( $_REQUEST['ywacp_is_single'] ) && ! $this->enable_loop ) ) {
				return $datas;
			}

			$variation_id = ! empty( $_REQUEST['variation_id'] ) ? absint( $_REQUEST['variation_id'] ) : 0;
			$product_id   = $variation_id ? $variation_id : absint( $_REQUEST['product_id'] );
			$product      = wc_get_product( $product_id );
			if ( ! $product ) {
				return $datas;
			}
			$product_id = yit_get_base_product_id( $product );
			// Check if is excluded.
			$exclusion_only = 'exclusion-list' === get_option( 'yith-wacp-exclusion', yith_wacp_get_default( 'exclusion' ) );

			if ( ( $this->is_in_exclusion( $product_id ) && ! $exclusion_only ) || ( $exclusion_only && ! $this->is_in_exclusion( $product_id ) ) ) {
				return $datas;
			}

			$quantity                   = isset( $_REQUEST['quantity'] ) ? absint( $_REQUEST['quantity'] ) : 1;
			$layout                     = get_option( 'yith-wacp-layout-popup', yith_wacp_get_default( 'layout-popup' ) );
			$datas['yith_wacp_message'] = $this->get_popup_content( $product, $layout, $quantity );
			// phpcs:enable WordPress.Security.NonceVerification.Recommended

			/**
			 * APPLY_FILTERS: yith_wacp_add_to_cart_success_data
			 *
			 * Handles added to cart success data.
			 *
			 * @param array      $datas    Add to cart data.
			 * @param WC_Product $product  The added product.
			 * @param string     $layout   The popup layout.
			 * @param integer    $quantity The product quantity.
			 */
			return apply_filters( 'yith_wacp_add_to_cart_success_data', $datas, $product, $layout, $quantity );
		}

		/**
		 * Action ajax for remove item from cart
		 *
		 * @access public
		 * @since  1.0.0
		 */
		public function remove_item_cart_ajax() {

			// phpcs:disable WordPress.Security.NonceVerification.Recommended
			if ( ! isset( $_REQUEST['action'] ) || $_REQUEST['action'] !== $this->action_remove || ! isset( $_REQUEST['item_key'] ) ) {
				die();
			}

			$item_key = sanitize_text_field( wp_unslash( $_REQUEST['item_key'] ) );

			WC()->cart->remove_cart_item( $item_key );

			if ( ! WC()->cart->is_empty() ) {
				$cart    = WC()->cart->get_cart();
				$last    = end( $cart );
				$product = isset( $last['product_id'] ) ? wc_get_product( $last['product_id'] ) : false;

				// Remove popup message.
				remove_action( 'yith_wacp_before_popup_content', array( $this, 'add_message' ), 10 );
				$template = 'cart';
			} else {
				$product  = null;
				$template = 'empty';
			}

			$return = $this->get_popup_content( $product, $template );

			// phpcs:enable WordPress.Security.NonceVerification.Recommended
			wp_send_json(
				array(
					'html'  => $return,
					'items' => ! is_null( WC()->cart ) ? WC()->cart->get_cart_contents_count() : 0,
				)
			);
		}

		/**
		 * Action ajax for update item from cart
		 *
		 * @access public
		 * @since  1.3.0
		 */
		public function update_item_cart_ajax() {

			// phpcs:disable WordPress.Security.NonceVerification.Recommended
			if ( ! isset( $_REQUEST['action'] ) || $_REQUEST['action'] !== $this->action_update || ! isset( $_REQUEST['item_key'] ) ) {
				die();
			}

			$item_key = sanitize_text_field( wp_unslash( $_REQUEST['item_key'] ) );
			$qty      = apply_filters( 'woocommerce_stock_amount_cart_item', wc_stock_amount( isset( $_REQUEST['qty'] ) ? absint( $_REQUEST['qty'] ) : 1 ), $item_key );
			$item     = WC()->cart->get_cart_item( $item_key );

			if ( ! empty( $item ) ) {
				$passed_validation = apply_filters( 'woocommerce_update_cart_validation', true, $item_key, $item, $qty );
				// If validation passed -> set new quantity.
				$passed_validation && WC()->cart->set_quantity( $item_key, $qty, true );
			} else {
				$cart = WC()->cart->get_cart();
				$item = end( $cart );
			}

			$product = false;
			if ( ! WC()->cart->is_empty() ) {
				$product  = isset( $item['product_id'] ) ? wc_get_product( $item['product_id'] ) : false;
				$template = 'cart';
			} else {
				$template = 'empty';
			}

			// Remove popup message.
			remove_action( 'yith_wacp_before_popup_content', array( $this, 'add_message' ), 10 );
			add_action( 'yith_wacp_before_popup_content', 'wc_print_notices', 10 );

			// phpcs:enable WordPress.Security.NonceVerification.Recommended
			wp_send_json(
				array(
					'html'  => $this->get_popup_content( $product, $template ),
					'items' => ! is_null( WC()->cart ) ? WC()->cart->get_cart_contents_count() : 0,
				)
			);
		}

		/**
		 * Action ajax for add to cart in single product page
		 *
		 * @access public
		 * @since  1.0.0
		 */
		public function add_item_cart_ajax() {

			// phpcs:disable WordPress.Security.NonceVerification.Recommended
			if ( ! isset( $_REQUEST['action'] ) || $_REQUEST['action'] !== $this->action_add || ! isset( $_REQUEST['product_id'] ) ) {
				return;
			}

			$product_id = absint( $_REQUEST['product_id'] );
			// Get WooCommerce error notice.
			$error_notices = wc_get_notices( 'error' );

			if ( ! $this->last_cart_item && function_exists( 'wc_gfpa' ) && method_exists( wc_gfpa(), 'get_gravity_form_data' ) ) {
				// Check if we need a gravity form!
				$context           = ! empty( $_POST['add-variations-to-cart'] ) ? 'bulk' : 'single'; // phpcs:ignore WordPress.Security.NonceVerification.Missing
				$gravity_form_data = wc_gfpa()->get_gravity_form_data( $product_id, $context );
				if ( ! empty( $gravity_form_data ) ) {
					$error_notices = array( esc_html__( 'There was a problem with your submission. Please check the required fields below.', 'yith-woocommerce-added-to-cart-popup' ) );
				}
			}

			if ( $error_notices ) {
				// Print notices if any!
				ob_start();
				foreach ( $error_notices as $value ) {
					$value = ( is_array( $value ) && isset( $value['notice'] ) ) ? $value['notice'] : $value;
					$value && wc_print_notice( $value, 'error' );
				}
				$error = ob_get_clean();

				wc_clear_notices();

				wp_send_json( array( 'error' => $error ) );

			} else {
				// Clear other notices.
				wc_clear_notices();

				// Trigger action for added to cart in AJAX.
				do_action( 'woocommerce_ajax_added_to_cart', $product_id );
				WC_AJAX::get_refreshed_fragments();
			}

		}

		/**
		 * Action ajax for frequently bought together form
		 *
		 * @access public
		 *
		 * @param array      $products_added An array of products added.
		 * @param WC_Product $main_product   The main product object.
		 * @param array      $offered        An array of offered products.
		 *
		 * @since  1.4.0
		 */
		public function add_wfbt_cart_ajax( $products_added, $main_product, $offered ) {

			if ( ! isset( $_REQUEST['_wpnonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_REQUEST['_wpnonce'] ) ), 'yith_bought_together' ) || ! isset( $_REQUEST['actionAjax'] ) || sanitize_text_field( wp_unslash( $_REQUEST['actionAjax'] ) ) !== $this->action_add_wfbt ) {
				return;
			}

			// Get WooCommerce error notices!
			$error_notices = wc_get_notices( 'error' );
			if ( $error_notices ) {
				ob_start();
				foreach ( $error_notices as $value ) {
					$value = ( is_array( $value ) && isset( $value['notice'] ) ) ? $value['notice'] : $value;
					$value && wc_print_notice( $value, 'error' );
				}
				$error = ob_get_clean();

				wp_send_json( array( 'error' => $error ) );
			} else {
				$product = wc_get_product( $main_product );
				if ( $product->is_type( 'variation' ) ) {
					$product = wc_get_product( $product->get_parent_id() );
				}
				$popup_html = $this->get_popup_content( $product, 'cart' );

				// Clear other notices.
				wc_clear_notices();

				wp_send_json(
					array(
						'html'  => $popup_html,
						'items' => ! is_null( WC()->cart ) ? WC()->cart->get_cart_contents_count() : 0,
					)
				);
			}
		}

		/**
		 * Prevent url redirect in add to cart ajax for single product page
		 *
		 * @param string $url The url destination after add to cart action in single product page.
		 *
		 * @return boolean | string
		 * @since  1.0.0
		 */
		public function prevent_redirect_url( $url ) {
			// phpcs:ignore WordPress.Security.NonceVerification.Recommended
			if ( ! empty( $_REQUEST['action'] ) && $_REQUEST['action'] === $this->action_add ) {
				return false;
			}

			return $url;
		}

		/**
		 * Add message before popup content
		 *
		 * @access public
		 *
		 * @param WC_Product $product The product object.
		 *
		 * @since  1.0.0
		 */
		public function add_message( $product ) {
			$message       = get_option( 'yith-wacp-popup-message', yith_wacp_get_default( 'popup-message' ) );
			$use_icon      = get_option( 'yith-wacp-notice-icon', yith_wacp_get_default( 'notice-icon' ) );
			$message_style = get_option( 'yith-wacp-popup-message-style', yith_wacp_get_default( 'popup-message-style' ) );

			if ( ! $message || ! $product ) {
				return;
			}

			ob_start();
			?>
			<div class="yith-wacp-message <?php echo esc_attr( $use_icon ); ?>">
				<?php echo wp_kses_post( sprintf( '<%1$s>%2$s</%1$s>', $message_style, $message ) ); ?>
			</div>
			<?php

			/**
			 * APPLY_FILTERS: yith_wacp_message_popup_html
			 *
			 * Handles added to cart message HTML.
			 *
			 * @param string     $html    Added to cart message HTML.
			 * @param WC_Product $product The added product.
			 */
			echo wp_kses_post( apply_filters( 'yith_wacp_message_popup_html', ob_get_clean(), $product ) );
		}

		/**
		 * Add action button to popup
		 *
		 * @access public
		 *
		 * @param WC_Product $product The product object.
		 *
		 * @since  1.0.0
		 */
		public function add_actions_button( $product ) {

			$cart           = get_option( 'yith-wacp-show-go-cart', yith_wacp_get_default( 'show-go-cart' ) ) === 'yes';
			$checkout       = get_option( 'yith-wacp-show-go-checkout', yith_wacp_get_default( 'show-go-checkout' ) ) === 'yes';
			$continue       = get_option( 'yith-wacp-show-continue-shopping', yith_wacp_get_default( 'show-continue-shopping' ) ) === 'yes';
			$active_buttons = count( array_filter( array( $cart, $checkout, $continue ) ) );

			if ( apply_filters( 'yith_wacp_check_active_buttons_display', true ) && ( $active_buttons < 1 || WC()->cart->is_empty() ) ) {
				return;
			}

			$args = array(
				'cart'                  => $cart,
				/**
				 * APPLY_FILTERS: yith_wacp_go_cart_url
				 *
				 * Handles "Go to cart" button URL.
				 *
				 * @param string $url Button URL.
				 */
				'cart_url'              => apply_filters( 'yith_wacp_go_cart_url', wc_get_page_permalink( 'cart' ) ),
				'checkout'              => $checkout,
				/**
				 * APPLY_FILTERS: yith_wacp_go_checkout_url
				 *
				 * Handles "Go to checkout" button URL.
				 *
				 * @param string $url Button URL.
				 */
				'checkout_url'          => apply_filters( 'yith_wacp_go_checkout_url', wc_get_checkout_url() ),
				'continue'              => $continue,
				/**
				 * APPLY_FILTERS: yith_wacp_continue_shopping_url
				 *
				 * Handles "Continue shopping" button URL.
				 *
				 * @param string     $url     Button URL.
				 * @param WC_Product $product The added product.
				 */
				'continue_shopping_url' => apply_filters( 'yith_wacp_continue_shopping_url', '#', $product ),
				'layout'                => "layout-$active_buttons-buttons",
			);

			ob_start();
			wc_get_template( 'yith-wacp-popup-action.php', $args, '', YITH_WACP_TEMPLATE_PATH . '/' );

			/**
			 * APPLY_FILTERS: yith_wacp_actions_button_html
			 *
			 * Handles action buttons HTML.
			 *
			 * @param string     $html    Action buttons HTML.
			 * @param WC_Product $product The added product.
			 */
			echo wp_kses_post( apply_filters( 'yith_wacp_actions_button_html', ob_get_clean(), $product ) );
		}

		/**
		 * Add suggested/related product to popup
		 *
		 * @access public
		 *
		 * @param WC_Product $product The product object.
		 *
		 * @since  1.1.0
		 */
		public function add_related( $product ) {

			if ( 'yes' !== get_option( 'yith-wacp-show-related', yith_wacp_get_default( 'show-related' ) ) ) {
				return;
			}

			$suggested           = get_option( 'yith-wacp-related-products', yith_wacp_get_default( 'related-products' ) );
			$number_of_suggested = get_option( 'yith-wacp-related-number', yith_wacp_get_default( 'related-number' ) );
            $number_of_suggested = absint( $number_of_suggested );
			$suggested_type      = get_option( 'yith-wacp-suggested-products-type', yith_wacp_get_default( 'suggested-products-type' ) );
			$product_id          = $product ? yit_get_base_product_id( $product ) : 0;

			if ( ! empty( $suggested ) && 'specific_products' === $suggested_type ) {
				if ( ! is_array( $suggested ) ) {
					$suggested = explode( ',', $suggested );
				}
				$suggested = array_filter( $suggested ); // Remove empty if any.
			} elseif ( $product ) {

				$product_id = yit_get_base_product_id( $product );
				// Get correct product if original is variation.
				if ( $product instanceof WC_Product && $product->is_type( 'variation' ) ) {
					$product = wc_get_product( $product_id );
				}

				switch ( $suggested_type ) {
					case 'crossell':
						$suggested = WC()->cart->get_cross_sells();
						break;
					case 'upsell':
						$suggested = $product->get_upsell_ids();
						break;
					case 'related':
						$suggested = wc_get_related_products( $product_id, $number_of_suggested );
						break;
					default:
						$suggested = wc_get_related_products( $product_id, $number_of_suggested );
						break;
				}
			}

			$suggested = $this->filter_product_already_in_cart( $suggested );

			if ( empty( $suggested ) ) {
				return;
			}

			/**
			 * APPLY_FILTERS: yith_wacp_related_products_args
			 *
			 * Handles related product query args.
			 *
			 * @param array $args Related proucts query arguments.
			 */
			$query_args = apply_filters(
				'yith_wacp_related_products_args',
				array(
					'post_type'           => array( 'product', 'product_variation' ),
					'post_status'         => 'publish',
					'ignore_sticky_posts' => 1,
					'no_found_rows'       => 1,
					'posts_per_page'      => $number_of_suggested,
					'post__in'            => $suggested,
					'post__not_in'        => array( $product_id ),
					'orderby'             => 'rand',
				),
				$product_id
			);

			$products = new WP_Query( $query_args );
			if ( $products->have_posts() ) {
				/**
				 * APPLY_FILTERS: yith_wacp_popup_related_args
				 *
				 * Handles related product display args.
				 *
				 * @param array $args Related proucts display arguments.
				 */
				$args = apply_filters(
					'yith_wacp_popup_related_args',
					array(
						'products'         => $products,
						'title'            => get_option( 'yith-wacp-related-title', yith_wacp_get_default( 'related-title' ) ),
						'columns'          => get_option( 'yith-wacp-related-columns', yith_wacp_get_default( 'related-columns' ) ),
						'show_add_to_cart' => get_option( 'yith-wacp-suggested-add-to-cart', yith_wacp_get_default( 'suggested-add-to-cart' ) ) === 'yes',
					),
					$product
				);

				wc_get_template( 'yith-wacp-popup-related.php', $args, '', YITH_WACP_TEMPLATE_PATH . '/' );
			}
			wp_reset_postdata();

		}

		/**
		 * From given product array remove products already in cart
		 *
		 * @param array $products An array of products to show in popup.
		 *
		 * @return array
		 * @since  1.5.5
		 */
		public function filter_product_already_in_cart( $products ) {
			if ( empty( $products ) ) {
				return array();
			}

			$products = array_unique( $products );
			$in_cart  = array();
			foreach ( WC()->cart->get_cart() as $cart_item_key => $values ) {
				if ( $values['quantity'] > 0 ) {
					$in_cart[] = $values['product_id'];
				}
			}

            /**
             * APPLY_FILTERS: yith_wacp_filter_product_already_in_cart
             *
             * Filter the products if they are already in the cart.
             *
             * @param array $product_diff Compute the difference of both arrays.
             * @param array $products     The products.
             * @param array $in_cart      The products already in the cart.
             */
			return apply_filters( 'yith_wacp_filter_product_already_in_cart', array_diff( $products, $in_cart ), $products, $in_cart );
		}

		/**
		 * Exclude product from added to cart popup process in single product page
		 *
		 * @since  1.0.0
		 */
		public function exclude_single() {
			global $product;

			$product_id     = yit_get_base_product_id( $product );
			$exclusion_only = 'exclusion-list' === get_option( 'yith-wacp-exclusion', yith_wacp_get_default( 'exclusion' ) );

			if ( ( $this->is_in_exclusion( $product_id ) && ! $exclusion_only ) || ( $exclusion_only && ! $this->is_in_exclusion( $product_id ) ) ) {
				echo '<input type="hidden" name="yith-wacp-is-excluded" value="1" />';
			}
		}

		/**
		 * Check if product is in exclusion list
		 *
		 * @param integer $product_id The product ID to check.
		 *
		 * @return boolean
		 * @since  1.0.0
		 */
		public function is_in_exclusion( $product_id ) {

			$is_in_exclusion    = false;
			$exclusion_prod     = array_filter( explode( ',', get_option( 'yith-wacp-exclusions-prod-list', '' ) ) );
			$exclusion_cat      = array_filter( explode( ',', get_option( 'yith-wacp-exclusions-cat-list', '' ) ) );
			$product_cat        = array();
			$product_categories = get_the_terms( $product_id, 'product_cat' );

			if ( ! empty( $product_categories ) ) {
				foreach ( $product_categories as $cat ) {
					$product_cat[] = $cat->term_id;
				}
			}

			$intersect = array_intersect( $product_cat, $exclusion_cat );
			if ( in_array( $product_id, $exclusion_prod ) || ! empty( $intersect ) ) { // phpcs:ignore
				$is_in_exclusion = true;
			}

			/**
			 * APPLY_FILTERS: yith_wacp_is_in_exclusion
			 *
			 * Handles product exclusion.
			 *
			 * @param boolean $is_in_exclusion Check if product is excluded.
			 * @param integer $product_id      The product ID.
			 */
			return apply_filters( 'yith_wacp_is_in_exclusion', $is_in_exclusion, $product_id );
		}

		/**
		 * Prevent cart redirect. WC Option Redirect to the cart page after successful addition
		 *
		 * @param mixed  $value  Current value.
		 * @param string $option Option name.
		 *
		 * @return mixed
		 * @since  1.1.0
		 */
		public function prevent_cart_redirect( $value, $option ) {
			if ( ( is_product() && $this->enable_single ) || $this->enable_loop ) {
				return 'no';
			}

			return $value;
		}

		/**
		 * Store last cart item key
		 *
		 * @param string         $cart_item_key  The cart item key.
		 * @param string|integer $product_id     The product object ID.
		 * @param string|integer $quantity       Quantity value.
		 * @param string|integer $variation_id   The variation object ID.
		 * @param array          $variation      The variation data.
		 * @param array          $cart_item_data The cart item data.
		 *
		 * @since  1.1.1
		 */
		public function store_cart_item_key( $cart_item_key, $product_id, $quantity, $variation_id, $variation, $cart_item_data ) {

			if ( in_array( '_wjecf_free_product_coupon', $cart_item_data ) ) { // phpcs:ignore
				return;
			}

			$this->last_cart_item = $cart_item_key;
		}

		/**
		 * Filter item data and add item
		 *
		 * @param string     $value   Current value.
		 * @param WC_Product $product The product instance.
		 *
		 * @return string
		 * @since  1.2.1
		 */
		public function filter_get_name_product( $value, $product ) {
			$id = $product->get_id();

			// phpcs:ignore WordPress.Security.NonceVerification.Recommended
			if ( isset( $_REQUEST['ywacp_is_single'] ) && 'yes' === sanitize_text_field( wp_unslash( $_REQUEST['ywacp_is_single'] ) ) && isset( $_REQUEST['variation_id'] ) && absint( $_REQUEST['variation_id'] ) === absint( $id ) && 'product' === get_option( 'yith-wacp-layout-popup', yith_wacp_get_default( 'layout-popup' ) ) ) {
				// Return parent name.
				return $product->get_title();
			}

			return $value;
		}

		/**
		 * Add param to related query args
		 *
		 * @param array $args Add tax query to related query.
		 *
		 * @return array
		 * @since  1.2.2
		 */
		public function add_related_query_args( $args ) {
			$args['tax_query'] = WC()->query->get_tax_query(); // phpcs:ignore

			return $args;
		}

		/**
		 * Add cart info (total, discount, shipping..) to popup template
		 *
		 * @since  1.4.0
		 */
		public function add_cart_info() {
			$cart_total    = get_option( 'yith-wacp-show-cart-totals', yith_wacp_get_default( 'show-cart-totals' ) ) === 'yes';
			$cart_shipping = get_option( 'yith-wacp-show-cart-shipping', yith_wacp_get_default( 'show-cart-shipping' ) ) === 'yes';
			$cart_tax      = get_option( 'yith-wacp-show-cart-tax', yith_wacp_get_default( 'show-cart-tax' ) ) === 'yes';

			if ( ! $cart_total && ! $cart_shipping && ! $cart_tax ) {
				return;
			}

			wc_get_template(
				'yith-wacp-popup-cart-info.php',
				array(
					'cart_info'     => yith_wacp_get_cart_info(),
					'cart_total'    => $cart_total,
					'cart_shipping' => $cart_shipping,
					'cart_tax'      => $cart_tax,
				),
				'',
				YITH_WACP_TEMPLATE_PATH . '/'
			);
		}

		/**
		 * Stop process discount for YITH Dynamic Pricing and Discount plugin
		 *
		 * @param boolean $value Current filter value.
		 *
		 * @return boolean
		 * @since  1.7.0
		 */
		public function ywdpd_stop_process_discount( $value ) {
			if ( isset( $_REQUEST['action'] ) ) {
				$action = sanitize_text_field( wp_unslash( $_REQUEST['action'] ) );
				if ( $this->action_update === $action || $this->action_remove === $action ) {
					return true;
				}
			}

			return $value;
		}

		/**
		 * Check if is single product page add to cart and set ajax request to true for YITH Multi Currency Switcher
		 *
		 * @param boolean $is_ajax Current filter value. Tru eif is an ajax request, false otherwise.
		 *
		 * @return boolean
		 * @since  1.10.0
		 */
		public function set_multi_currency_ajax_request( $is_ajax ) {
			return $is_ajax || isset( $_REQUEST['ywacp_is_single'] );
		}
	}
}
