<?php
/**
 * Mini Cart class
 *
 * @package YITH\AddedToCartPopup\Frontend
 */

defined( 'YITH_WACP' ) || exit; // Exit if accessed directly.

if ( ! class_exists( 'YITH_WACP_Mini_Cart' ) ) {
	/**
	 * Mini Cart class
	 * The class manage mini cart feature behaviors.
	 *
	 * @class   YITH_WACP_Mini_Cart
	 * @since   1.0.0
	 * @author  YITH <plugins@yithemes.com>
	 * @package YITH\AddedToCartPopup\Frontend
	 */
	class YITH_WACP_Mini_Cart {

		use YITH_WACP_Trait_Singleton;

		/**
		 * Action update mini cart
		 *
		 * @since 1.4.0
		 * @var string
		 */
		public $action_update_cart = 'yith_wacp_update_mini_cart';

		/**
		 * Constructor
		 *
		 * @access public
		 * @since  1.0.0
		 */
		public function __construct() {

			add_action( 'wc_ajax_' . $this->action_update_cart, array( $this, 'update_mini_cart_ajax' ) );
			add_action( 'wp_ajax_nopriv' . $this->action_update_cart, array( $this, 'update_mini_cart_ajax' ) );
			add_filter( 'yith_wacp_frontend_script_localized_args', array( $this, 'add_mini_cart_args' ), 10, 1 );
			add_filter( 'yith_wacp_add_to_cart_success_data', array( $this, 'add_data_to_response' ), 10, 4 );
			add_action( 'wp_footer', array( $this, 'add_mini_cart_template' ) );
		}

		/**
		 * Add mini data cart to response
		 *
		 * @param array          $data     An array of popup response data.
		 * @param WC_Product     $product  The product object.
		 * @param string         $layout   Popup layout.
		 * @param integer|string $quantity Product quantity.
		 *
		 * @return array
		 * @since  1.4.0
		 */
		public function add_data_to_response( $data, $product, $layout, $quantity ) {
			$data['yith_wacp_message_cart'] = 'product' !== $layout ? $data['yith_wacp_message'] : YITH_WACP_Frontend::get_instance()->get_popup_content( $product, 'cart', $quantity );
			$data['yith_wacp_cart_items']   = ! is_null( WC()->cart ) ? WC()->cart->get_cart_contents_count() : 0;

			return $data;
		}

		/**
		 * Add arguments to localized array
		 *
		 * @param array $args Localized array.
		 *
		 * @return array
		 * @since  1.4.0
		 */
		public function add_mini_cart_args( $args ) {
			$args['actionUpdateMiniCart'] = $this->action_update_cart;
			$args['mini_cart_position']   = get_option(
				'yith-wacp-mini-cart-position',
				array(
					'top'  => 20,
					'left' => 97,
				)
			);

			return $args;
		}

		/**
		 * Update mini cart using Ajax
		 *
		 * @since  1.4.0
		 */
		public function update_mini_cart_ajax() {
			// phpcs:ignore WordPress.Security.NonceVerification.Recommended
			if ( ! isset( $_REQUEST['action'] ) || $_REQUEST['action'] !== $this->action_update_cart ) {
				die();
			}

			// Make sure to remove added message.
			remove_action( 'yith_wacp_before_popup_content', array( YITH_WACP_Frontend::get_instance(), 'add_message' ), 10 );
			// Get last product on cart and set related products.
			$product = false;
			if ( ! WC()->cart->is_empty() ) {
				$contents = WC()->cart->get_cart_contents();
				$item     = end( $contents );
				$product  = isset( $item['product_id'] ) ? wc_get_product( $item['product_id'] ) : false;
				$template = 'cart';
			} else {
				$template = 'empty';
			}

			wp_send_json(
				array(
					'html'  => YITH_WACP_Frontend::get_instance()->get_popup_content( $product, $template ),
					'items' => ! is_null( WC()->cart ) ? WC()->cart->get_cart_contents_count() : 0,
				)
			);
		}

		/**
		 * Add mini cart template
		 *
		 * @since  1.4.0
		 */
		public function add_mini_cart_template() {

			if ( apply_filters( 'yith_wacp_hide_mini_cart', true ) && ( is_cart() || is_checkout() ) ) {
				return;
			}

			wc_get_template(
				'yith-wacp-mini-cart.php',
				array(
					'items'        => ! is_null( WC()->cart ) ? WC()->cart->get_cart_contents_count() : 0,
					'icon_type'    => get_option( 'yith-wacp-mini-cart-icon-type', yith_wacp_get_default( 'mini-cart-icon-type' ) ) . '-icon',
					'icon'         => get_option( 'yith-wacp-mini-cart-icon', yith_wacp_get_default( 'mini-cart-icon' ) ),
					'show_counter' => get_option( 'yith-wacp-mini-cart-show-counter', yith_wacp_get_default( 'mini-cart-show-counter' ) ) === 'yes',
					'hide_empty'   => get_option( 'yith-wacp-mini-cart-hide-empty', yith_wacp_get_default( 'mini-cart-hide-empty' ) ) === 'yes' ? 'hide-if-empty' : '',
				),
				'',
				YITH_WACP_TEMPLATE_PATH . '/'
			);
		}
	}

}
