<?php
/**
 * YITH_YWBC_Shortcodes class
 *
 * @package YITH\Barcodes\Classes
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

if ( ! class_exists( 'YITH_YWBC_Shortcodes' ) ) {

	/**
	 * YITH_YWBC_Shortcodes
	 *
	 * @class   YITH_YWBC_Shortcodes
	 * @package Yithemes
	 * @since   1.0.0
	 * @author  YITH <plugins@yithemes.com>
	 */
	class YITH_YWBC_Shortcodes {

		/**
		 * Single instance of the class
		 *
		 * @since 1.0.0
		 * @var instance
		 */
		protected static $instance;
		/**
		 * Get_instance
		 *
		 * @return instance
		 */
		public static function get_instance() {
			if ( is_null( self::$instance ) ) {
				self::$instance = new self();
			}

			return self::$instance;
		}

		/**
		 * Constructor
		 *
		 * Initialize plugin and registers actions and filters to be used
		 *
		 * @since  1.0
		 */
		protected function __construct() {

			/**
			 * Add a shortcode for rendering a barcode
			 */
			add_shortcode(
				'yith_render_barcode',
				array(
					$this,
					'yith_render_barcode',
				)
			);

			add_shortcode(
				'yith_render_post_barcode',
				array(
					$this,
					'yith_render_post_barcode',
				)
			);

			/**
			 * Add a shortcode for searching for a order barcode value
			 */
			add_shortcode(
				'yith_order_barcode',
				array(
					$this,
					'yith_order_barcode',
				)
			);

			/**
			 * Add a shortcode for searching for a product barcode value
			 */
			add_shortcode(
				'yith_product_barcode',
				array(
					$this,
					'yith_product_barcode',
				)
			);

			/**
			 * Manage a search barcode value for products from the form generated by the shortcode
			 */
			add_filter(
				'yith_barcode_action_product_search',
				array(
					$this,
					'yith_barcode_action_product_search',
				),
				10,
				2
			);

			/**
			 * Manage a search barcode value for products from the form generated by the shortcode
			 */
			add_filter(
				'yith_barcode_action_shop_order_complete_order',
				array(
					$this,
					'yith_barcode_action_shop_order_complete_order',
				),
				10,
				2
			);

			/**
			 * Manage a search barcode value for products from the form generated by the shortcode
			 */
			add_filter(
				'yith_barcode_action_shop_order_search',
				array(
					$this,
					'yith_barcode_action_shop_order_search',
				),
				10,
				2
			);

			/**
			 * Manage request from the form generated by the shortcode
			 */
			add_action(
				'wp_ajax_barcode_actions',
				array(
					$this,
					'manage_barcode_actions_callback',
				)
			);

			add_action(
				'wp_ajax_nopriv_barcode_actions',
				array(
					$this,
					'manage_barcode_actions_callback',
				)
			);

			/**
			 * Manage request from the form generated by the shortcode for the orders
			 */
			add_action(
				'wp_ajax_barcode_actions_search_shop_order',
				array(
					$this,
					'manage_barcode_search_callback',
				)
			);

			add_action(
				'wp_ajax_nopriv_barcode_actions_search_shop_order',
				array(
					$this,
					'manage_barcode_search_callback',
				)
			);

			add_action(
				'wp_ajax_apply_barcode_action',
				array(
					$this,
					'set_barcode_status',
				)
			);

			add_action(
				'wp_ajax_nopriv_apply_barcode_action',
				array(
					$this,
					'set_barcode_status',
				)
			);

			add_action(
				'wp_ajax_change_stock_input_value',
				array(
					$this,
					'update_product_stock_callback',
				)
			);

			add_action(
				'wp_ajax_nopriv_change_stock_input_value',
				array(
					$this,
					'update_product_stock_callback',
				)
			);

			/**
			 * Manage request from the form generated by the shortcode for the orders
			 */
			add_action(
				'wp_ajax_barcode_actions_search_product',
				array(
					$this,
					'manage_barcode_search_callback',
				)
			);

			add_action(
				'wp_ajax_nopriv_barcode_actions_search_product',
				array(
					$this,
					'manage_barcode_search_callback',
				)
			);
		}

		/**
		 * Set_barcode_status
		 *
		 * @return void
		 */
		public function set_barcode_status() {
			if ( ! isset( $_POST['id'] ) || ! isset( $_POST['status'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification
				die();
			}

			$id              = intval( sanitize_text_field( wp_unslash( $_POST['id'] ) ) ); // phpcs:ignore WordPress.Security.NonceVerification
			$status          = sanitize_text_field( wp_unslash( $_POST['status'] ) ); // phpcs:ignore WordPress.Security.NonceVerification
			$type            = ( isset( $_POST['type'] ) ) ? sanitize_text_field( wp_unslash( $_POST['type'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification
			$barcode_actions = ( isset( $_POST['barcode-actions'] ) ) ? array_map( 'sanitize_text_field', wp_unslash( $_POST['barcode-actions'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification
			$store_id        = isset( $_POST['store_id'] ) ? $_POST['store_id'] : 0; // phpcs:ignore WordPress.Security.NonceVerification, WordPress.Security.ValidatedSanitizedInput

			// This callback is shared among all barcode search function
			// Set some value based on the object type.
			$template = '';
			$object   = null;
			if ( 'order' === $type ) {
				$object = wc_get_order( $id );
				if ( $object ) {
					if ( 'check-in-ticket' === $status && defined( 'YITH_WCEVTI_PREMIUM' ) ) {
						yith_wcevti_check_all_order_tickets_in( $id );
					}

					/** APPLY_FILTERS: yith_wcbc_before_set_barcode_status_for_order_condition
					 *
					 * Filter the condition to update or not the order status using the shortcode buttons.
					 *
					 * @param bool true to update the order status, false to not
					 * @param int $id the order ID.
					 */
					if ( apply_filters( 'yith_wcbc_before_set_barcode_status_for_order_condition', true, $id ) ) {
						$status_result = $object->set_status( $status );
						$object->save();

						$code   = $status_result['to'] === $status;
						$result = array(
							'code' => $code ? 1 : -1,
						);
					}

					/** DO_ACTION: yith_wcbc_set_barcode_status_for_order
					*
					* Adds an action when setting the barcode status for the order.
					*
					* @param $id              $_POST['id'].
					* @param $status          $_POST['status'].
					* @param $type            $_POST['type'].
					* @param $barcode_actions $_POST['barcode-actions'].
					*/
					do_action( 'yith_wcbc_set_barcode_status_for_order', $id, $status, $type, $barcode_actions );

				}

				$template = 'shortcode/ywbc-search-orders-row.php';
			} elseif ( 'product' === $type ) {
				$object   = wc_get_product( $id );
				$template = 'shortcode/ywbc-search-products-row.php';

				if ( ( 'increase-stock' === $status ) || ( 'decrease-stock' === $status ) ) {

					$stock_unit = isset( $_POST['stock_unit'] ) ? intval( $_POST['stock_unit'] ) : 1; // phpcs:ignore WordPress.Security.NonceVerification

					/** APPLY_FILTERS: yith_wcbc_set_stock_unit
					*
					* Filter the stock number when using the buttons in the shortcode.
					*
					* @param $value  Default value of stock changer.
					* @param $status Default status behavior of the stock changer ('increase-stock' by default).
					*/
					$unit = apply_filters( 'yith_wcbc_set_stock_unit', ( 'increase-stock' === $status ) ? (int) $stock_unit : - (int) $stock_unit, $status );

					$multi_stock = $object->get_meta( '_yith_pos_multistock' );

					if ( isset( $multi_stock[ $store_id ] ) ) {

						$stock_unit = $multi_stock[ $store_id ];
						$new_stock  = $stock_unit + $unit;

						$multi_stock[ $store_id ] = $new_stock;
						$object->update_meta_data( '_yith_pos_multistock', $multi_stock );
						$object->save();
					} else {
						$stock_unit = $object->get_stock_quantity();
						$new_stock  = $stock_unit + $unit;
						$object->set_stock_quantity( $new_stock );
						$object->save();
					}
				}
				if ( ( 'add-to-cart' === $status ) ) {
					WC()->cart->add_to_cart( $id, 1 );
				}

				/** DO_ACTION: yith_wcbc_set_barcode_status_for_product
				*
				* Adds an action when setting the barcode status for the product.
				*
				* @param $id              $_POST['id'].
				* @param $status          $_POST['status'].
				* @param $type            $_POST['type'].
				* @param $barcode_actions $_POST['barcode-actions'].
				*/
				do_action( 'yith_wcbc_set_barcode_status_for_product', $id, $status, $type, $barcode_actions );
			}

			/** APPLY_FILTERS: yith_set_barcode_status_object
			*
			* Filter the object when changing the status in the barcode shortcode.
			*
			* @param $object Default object.
			*/
			$object = apply_filters( 'yith_set_barcode_status_object', $object );

			/** APPLY_FILTERS: yith_set_barcode_status_template
			*
			* Filter the templace when changing the status in the barcode shortcode.
			*
			* @param $template Default template.
			*/
			$template = apply_filters( 'yith_set_barcode_status_template', $template );

			/** APPLY_FILTERS: yith_wcbc_before_get_barcode_status_template_condition
			 *
			 * Filter the condition to get or not the template after the barcode status update.
			 *
			 * @param bool true to update the order status, false to not
			 * @param int $id the order ID.
			 */
			if ( apply_filters( 'yith_wcbc_before_get_barcode_status_template_condition', true, $id ) ) {
				ob_start();
				wc_get_template(
					$template,
					array(
						'object'          => $object,
						'barcode_actions' => $barcode_actions,
					),
					'',
					YITH_YWBC_TEMPLATES_DIR
				);

				$content = ob_get_clean();

				$result['code']  = 1;
				$result['items'] = $content;

				wp_send_json( $result );
			}
		}

		/**
		 * Update product stock from the input
		 *
		 * @return void
		 */
		public function update_product_stock_callback() {

			if ( ! isset( $_POST['id'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification
				die();
			}

			$id              = intval( sanitize_text_field( wp_unslash( $_POST['id'] ) ) ); // phpcs:ignore WordPress.Security.NonceVerification
			$store_id        = isset( $_POST['store_id'] ) ? $_POST['store_id'] : 0; // phpcs:ignore WordPress.Security.NonceVerification, WordPress.Security.ValidatedSanitizedInput
			$stock_unit = isset( $_POST['stock_unit'] ) ? intval( $_POST['stock_unit'] ) : 1; // phpcs:ignore WordPress.Security.NonceVerification

			$product = wc_get_product( $id );

			$multi_stock = $product->get_meta( '_yith_pos_multistock' );

			if ( isset( $multi_stock[ $store_id ] ) ) {
				$stock_unit               = $multi_stock[ $store_id ];
				$multi_stock[ $store_id ] = $stock_unit;
				$product->update_meta_data( '_yith_pos_multistock', $multi_stock );
				$product->save();
			} else {
				$product->set_stock_quantity( $stock_unit );
				$product->save();
			}

			$result['code']  = 1;

			wp_send_json( $result );

		}

		/**
		 * Manage_barcode_search_callback
		 *
		 * @return void
		 */
		public function manage_barcode_search_callback() {

			if ( ! isset( $_POST['text'] ) || ! isset( $_POST['barcode-actions'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification
				die();
			}

			$text                 = sanitize_text_field( wp_unslash( $_POST['text'] ) ); // phpcs:ignore WordPress.Security.NonceVerification
			$barcode_actions_post = array_map( 'sanitize_text_field', wp_unslash( $_POST['barcode-actions'] ) ); // phpcs:ignore WordPress.Security.NonceVerification

			$barcode_actions = array();
			foreach ( $barcode_actions_post as $barcode_action_key => $barcode_action ) {

				if ( ! empty( $barcode_action ) ) {
					$barcode_actions[ $barcode_action_key ] = $barcode_action;
				}
			}

			$result = array(
				'code'  => - 1,
				'value' => esc_html__( 'The selected action could not be performed', 'yith-woocommerce-barcodes' ),
			);

			// This callback is shared among all barcode search function
			// Set some value based on the object type.
			$current_filter = current_filter();
			$template       = '';

			if ( 'wp_ajax_barcode_actions_search_shop_order' === $current_filter || 'wp_ajax_nopriv_barcode_actions_search_shop_order' === $current_filter ) {
				$items = $this->barcode_action_search(
					$text,
					'shop_order'
				);

				$template = 'shortcode/ywbc-search-orders.php';
			} elseif ( 'wp_ajax_barcode_actions_search_product' === $current_filter || 'wp_ajax_nopriv_barcode_actions_search_product' === $current_filter ) {
				$items    = $this->barcode_action_search(
					$text,
					array( 'product', 'product_variation' )
				);
				$template = 'shortcode/ywbc-search-products.php';
			}

			/** APPLY_FILTERS: yith_barcode_ajax_callback_search_items
			*
			* Filter the items when searching with ajax in the barcode shortcode.
			*
			* @param $items $this->barcode_action_search.
			*/
			$items = apply_filters( 'yith_barcode_ajax_callback_search_items', $items );

			/** APPLY_FILTERS: yith_barcode_ajax_callback_search_template
			*
			* Filter the template when searching with ajax in the barcode shortcode.
			*
			* @param $template Default template.
			*/
			$template = apply_filters( 'yith_barcode_ajax_callback_search_template', $template );

			/** DO_ACTION: yith_barcode_before_load_template_barcode_actions
			*
			* Adds an action before loading the template of the barcode actions.
			*
			* @param $items $this->barcode_action_search.
			*/
			do_action( 'yith_barcode_before_load_template_barcode_actions', $items );

			foreach ( $items as $item ) {

				if ( is_object( $item ) && 'shop_order' === $item->post_type ) {

					$order = new WC_Order( $item->ID );

					$automatic_action = get_option( 'ywbc_automatic_order_status', 'no' );

					if ( 'processing' === $automatic_action ) {
						$order->update_status( 'processing' );
					}

					if ( 'completed' === $automatic_action ) {
						$order->update_status( 'completed', '', apply_filters( 'yith_ywbc_is_manual_order_update', false ) );
					}

					/** APPLY_FILTERS: yith_barcode_before_load_order_barcode_redirect
					*
					* Filter the condition to redirect before loading when searching in the barcode shortcode.
					*/
					if ( apply_filters( 'yith_barcode_before_load_order_barcode_redirect', false ) ) {

						$redirect_page           = get_site_url() . '/wp-admin/post.php?post=' . $item->ID . '&action=edit';
						$result['redirect_page'] = $redirect_page;
					}
				}

				if ( is_object( $item ) && 'product' === $item->post_type || 'product_variation' === $item->post_type ) {

					$product = wc_get_product( $item->ID );

					$automatic_action = get_option( 'ywbc_automatic_stock', 'no' );

					$multi_stock = $product->get_meta( '_yith_pos_multistock' );
					$store_id    = isset( $_POST['store_id'] ) ? $_POST['store_id'] : 0; // phpcs:ignore WordPress.Security.NonceVerification, WordPress.Security.ValidatedSanitizedInput

					if ( 'decrease' === $automatic_action ) {

						if ( isset( $multi_stock[ $store_id ] ) ) {

							$stock_unit = $multi_stock[ $store_id ];
							$new_stock  = $stock_unit - 1;

							$multi_stock[ $store_id ] = $new_stock;
							$product->update_meta_data( '_yith_pos_multistock', $multi_stock );
							$product->save();
						} else {

							/** APPLY_FILTERS: ywbc_stock_decrease_amount
							*
							* Filter the decrease stock behavior when using the barcode shortcode.
							*/
							wc_update_product_stock( $product, apply_filters( 'ywbc_stock_decrease_amount', 1 ), 'decrease' );
						}
					}

					if ( 'increase' === $automatic_action ) {

						if ( isset( $multi_stock[ $store_id ] ) ) {

							$stock_unit = $multi_stock[ $store_id ];
							$new_stock  = $stock_unit + 1;

							$multi_stock[ $store_id ] = $new_stock;
							$product->update_meta_data( '_yith_pos_multistock', $multi_stock );
							$product->save();
						} else {

							/** APPLY_FILTERS: ywbc_stock_increase_amount
							*
							* Filter the increase stock behavior when using the barcode shortcode.
							*/
							wc_update_product_stock( $product, apply_filters( 'ywbc_stock_increase_amount', 1 ), 'increase' );
						}
					}
				}
			}

			ob_start();
			wc_get_template(
				$template,
				array(
					'posts'           => $items,
					'barcode_actions' => $barcode_actions,
				),
				'',
				YITH_YWBC_TEMPLATES_DIR
			);

			$content = ob_get_clean();

			$result['code']  = 1;
			$result['items'] = $content;

			wp_send_json( $result );
		}

		/**
		 * Manage_barcode_actions_callback
		 *
		 * @return void
		 */
		public function manage_barcode_actions_callback() {

			if ( ! isset( $_POST['type'] ) || ! isset( $_POST['text'] ) || ! isset( $_POST['value'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification
				return;
			}
			$text   = sanitize_text_field( wp_unslash( $_POST['text'] ) ); // phpcs:ignore WordPress.Security.NonceVerification
			$action = sanitize_text_field( wp_unslash( $_POST['value'] ) ); // phpcs:ignore WordPress.Security.NonceVerification
			$type   = sanitize_text_field( wp_unslash( $_POST['type'] ) ); // phpcs:ignore WordPress.Security.NonceVerification

			$result = array(
				'code'  => - 1,
				'value' => esc_html__( 'The selected action could not be performed', 'yith-woocommerce-barcodes' ),
			);

			$result = apply_filters( 'yith_barcode_action_' . $type . '_' . $action, $result, $text );
			wp_send_json( $result );
		}

		/**
		 * Manage the shortcode 'yith_render_barcode' for rendering a barcode by object id or with specific value
		 *
		 * @param array $atts the shortcode attributes.
		 *
		 * @return string
		 */
		public function yith_render_barcode( $atts ) {

			$fields = shortcode_atts(
				array(
					'id'            => 0,
					'hide_if_empty' => 1,
					'value'         => '',
					'protocol'      => 'EAN8',
					'inline_css'    => '',
					'layout'        => '',

				),
				$atts
			);

			ob_start();

			// if id>0 show the barcode for a specific object.
			if ( $fields['id'] ) {
				YITH_YWBC()->show_barcode( $fields['id'], $fields['hide_if_empty'], $fields['inline_css'], $fields['layout'] );
			} elseif ( $fields['value'] ) {
				// Show barcode with specific value and protocol.
				$barcode  = new YITH_Barcode();
				$protocol = $fields['protocol'];
				$value    = $fields['value'];

				$barcode->generate( $protocol, $value );
				YITH_YWBC()->show_barcode( $barcode, $fields['hide_if_empty'], $fields['inline_css'], $fields['layout'] );
			}

			return ob_get_clean();
		}

		/**
		 * Yith_render_post_barcode
		 *
		 * @param  mixed $atts atts.
		 * @return show_barcode
		 */
		public function yith_render_post_barcode( $atts ) {

			global $post;

			$fields = shortcode_atts(
				array(
					'hide_if_empty' => 1,
					'inline_css'    => '',
					'layout'        => '',
				),
				$atts
			);

			ob_start();

			if ( $post->ID ) {
				YITH_YWBC()->show_barcode( $post->ID, $fields['hide_if_empty'], $fields['inline_css'], $fields['layout'] );
			}

			return ob_get_clean();
		}

		/**
		 * Manage the shortcode 'yith_order_barcode'
		 *
		 * @param array $atts the shortcode attributes.
		 *
		 * @return string
		 */
		public function yith_order_barcode( $atts ) {

			$processing_button = get_option( 'ywbc_order_shortcode_processing_button', 'yes' );
			$completed_button  = get_option( 'ywbc_order_shortcode_completed_button', 'yes' );

			$actions = array();

			if ( 'yes' === $processing_button ) {
				$actions['processing'] = __( 'Processing', 'yith-woocommerce-barcodes' );
			}

			if ( 'yes' === $completed_button ) {
				$actions['completed'] = __( 'Completed', 'yith-woocommerce-barcodes' );
			}

			/** APPLY_FILTERS: ywbc_order_shortcode_buttons_actions
			*
			* Filter the default actions in the order shortcode.
			*
			* @param $actions Default actions.
			*/
			$actions = apply_filters( 'ywbc_order_shortcode_buttons_actions', $actions );

			$fields = shortcode_atts(
				array(
					'search_type' => 'shop_order',
					'capability'  => get_option( 'ywbc_order_shortcode_capability_roles', array( 'administrator' ) ),
					'style'       => 'buttons',
					'actions'     => $actions,
				),
				$atts
			);

			ob_start();
			wc_get_template(
				'shortcode/ywbc-barcode-actions.php',
				array(
					'fields' => $fields,

					/** APPLY_FILTERS: yith_search_order_barcode_title
					*
					* Filter the Order barcode label in the order shortcode.
					*/
					'title'  => apply_filters(
						'yith_search_order_barcode_title',
						esc_html_x(
							'Order barcode',
							'title for product barcode form',
							'yith-woocommerce-barcodes'
						)
					),
				),
				'',
				YITH_YWBC_TEMPLATES_DIR
			);
			$content = ob_get_clean();

			return $content;
		}

		/**
		 * Manage the shortcode 'yith_product_barcode'
		 *
		 * @param array $atts the shortcode attributes.
		 *
		 * @return string
		 */
		public function yith_product_barcode( $atts ) {

			$stock_buttons      = get_option( 'ywbc_product_shortcode_stock_buttons', 'yes' );
			$add_to_cart_button = get_option( 'ywbc_product_shortcode_add_to_cart', 'no' );

			$actions = array();

			if ( 'yes' === $stock_buttons ) {
				$actions['increase-stock'] = true;
				$actions['decrease-stock'] = true;
			}

			if ( 'yes' === $add_to_cart_button ) {
				$actions['add-to-cart'] = true;
			}

			$fields = shortcode_atts(
				array(
					'search_type' => 'product',
					'capability'  => get_option( 'ywbc_product_shortcode_capability_roles', array( 'administrator' ) ),
					'style'       => 'buttons',
					'actions'     => $actions,
				),
				$atts
			);

			ob_start();
			wc_get_template(
				'shortcode/ywbc-barcode-actions.php',
				array(
					'fields' => $fields,

					/** APPLY_FILTERS: yith_search_product_barcode_title
					*
					* Filter the Product barcode label in the order shortcode.
					*/
					'title'  => apply_filters(
						'yith_search_product_barcode_title',
						esc_html_x(
							'Product barcode',
							'title for product barcode form',
							'yith-woocommerce-barcodes'
						)
					),
				),
				'',
				YITH_YWBC_TEMPLATES_DIR
			);

			$content = ob_get_clean();

			return $content;
		}


		/**
		 * Manage a search for barcode value on orders
		 *
		 * @param array  $result the result for the current action.
		 * @param string $text   the text entered by the user.
		 *
		 * @return string
		 * @since  1.0.0
		 */
		public function yith_barcode_action_shop_order_complete_order( $result, $text ) {
			$items = $this->barcode_action_search(
				$text,
				'shop_order'
			);

			$current_user = wp_get_current_user();
			if ( $current_user ) {
				/* translators: %s: Display name */
				$message = sprintf( esc_html__( 'Order status set by %s through YITH WooCommerce Barcodes and QR Codes.', 'yith-woocommerce-barcodes' ), $current_user->display_name );
			} else {
				$message = esc_html__( 'Order status set through YITH WooCommerce Barcodes and QR Codes.', 'yith-woocommerce-barcodes' );

			}
			foreach ( $items as $post ) {
				$order = wc_get_order( $post );
				if ( $order ) {
					$order->update_status( 'wc-completed', $message . '<BR>' );
				}
			}

			ob_start();
			wc_get_template(
				'shortcode/ywbc-complete-orders.php',
				array(
					'posts' => $items,
				),
				'',
				YITH_YWBC_TEMPLATES_DIR
			);

			$content = ob_get_clean();

			$result['code']  = 1;
			$result['value'] = $content;

			return $result;
		}

		/**
		 * Perform a search for a barcode value on specific object type
		 *
		 * @param string $text text.
		 * @param string $type type.
		 *
		 * @return array
		 * @since  1.0.0
		 */
		public function barcode_action_search( $text, $type = 'shop_order' ) {

			$args = array(
				'posts_per_page' => - 1,
				'post_type'      => $type,
				'post_status'    => 'any',
				'meta_query'     => array( //phpcs:ignore --Slow query
					array(
						'key'     => YITH_Barcode::YITH_YWBC_META_KEY_BARCODE_DISPLAY_VALUE,
						'value'   => $text,
						'compare' => 'LIKE',
					),
				),
			);

			/** APPLY_FILTERS: yith_ywbc_barcode_action_search
			*
			* Filter the search action in the shortcodes.
			*
			* @param array $args  Default plugin args.
			* @param string $text Default text.
			* @param string $type 'shop_order' by default.
			*/
			return apply_filters( 'yith_ywbc_barcode_action_search', get_posts( $args ), $args, $text, $type );

		}

		/**
		 * Manage a search for barcode value on orders
		 *
		 * @param array  $result the result for the current action.
		 * @param string $text   the text entered by the user.
		 *
		 * @since  1.0.0
		 */
		public function yith_barcode_action_shop_order_search( $result, $text ) {

		}
	}
}

YITH_YWBC_Shortcodes::get_instance();
