<?php
/**
 * YITH_YWBC_Backend class
 *
 * @package YITH\Barcodes\Classes
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

if ( ! class_exists( 'YITH_YWBC_Backend' ) ) {

	/**
	 * YITH_YWBC_Backend
	 *
	 * @class   YITH_YWBC_Backend
	 * @package Yithemes
	 * @since   1.0.0
	 * @author  YITH <plugins@yithemes.com>
	 */
	class YITH_YWBC_Backend {

		/**
		 * Single instance of the class
		 *
		 * @since 1.0.0
		 * @var instance
		 */
		protected static $instance;

		/**
		 * Get_instance
		 *
		 * @return instance
		 */
		public static function get_instance() {
			if ( is_null( self::$instance ) ) {
				self::$instance = new self();
			}

			return self::$instance;
		}

		/**
		 * Constructor
		 *
		 * Initialize plugin and registers actions and filters to be used
		 *
		 * @since  1.0
		 */
		protected function __construct() {
			$this->init_hooks();

			/** Gutenberg Support */
			add_action( 'plugins_loaded', array( $this, 'load_gutenberg_compatibility' ), 20 );

		}


		/**
		 * Initialize all hooks used by the plugin affecting the back-end behaviour
		 */
		public function init_hooks() {
			/**
			 * Enqueue scripts and styles for admin pages
			 */
			add_action(
				'admin_enqueue_scripts',
				array(
					$this,
					'enqueue_scripts',
				)
			);

			/**
			 * Enqueue scripts and styles for admin pages
			 */
			add_action(
				'admin_enqueue_scripts',
				array(
					$this,
					'enqueue_style',
				)
			);

			/**
			 * Create the barcode for orders when they are created
			 */
			add_action(
				'woocommerce_checkout_order_processed',
				array(
					$this,
					'on_new_order',
				)
			);

			if ( class_exists( 'WeDevs_Dokan' ) ) {
				add_action(
					'dokan_checkout_update_order_meta',
					array(
						$this,
						'on_new_order',
					)
				);
			}

			/**
			 * Create the barcode for products when they are created
			 */
			add_action(
				'transition_post_status',
				array(
					$this,
					'on_new_product',
				),
				10,
				3
			);

			/**
			 * Add the search by barcode value on back-end product list
			 */
			add_filter(
				'posts_join',
				array(
					$this,
					'query_barcode_join',
				),
				10,
				2
			);

			/**
			 * Add the search by barcode value on back-end product list
			 */
			add_filter(
				'posts_where',
				array(
					$this,
					'query_barcode_where',
				),
				10,
				2
			);

			/**
			 * Add the barcode fields for the order search
			 */
			add_filter(
				'woocommerce_shop_order_search_fields',
				array(
					$this,
					'search_by_order_fields',
				),
				10,
				2
			);

			/**
			 * Manage a request from product bulk actions
			 */
			add_action(
				'load-edit.php',
				array(
					$this,
					'generate_bulk_action',
				)
			);

			if ( function_exists( 'yith_plugin_fw_is_wc_custom_orders_table_usage_enabled' ) && yith_plugin_fw_is_wc_custom_orders_table_usage_enabled() ) {
				add_filter( 'bulk_actions-woocommerce_page_wc-orders', array( $this, 'generate_orders_bulk_action' ) );
				add_filter( 'handle_bulk_actions-woocommerce_page_wc-orders', array( $this, 'orders_bulk_actions_handler' ), 10, 3 );
			} else {
				add_filter( 'bulk_actions-edit-shop_order', array( $this, 'generate_orders_bulk_action' ) );
				add_filter( 'handle_bulk_actions-edit-shop_order', array( $this, 'orders_bulk_actions_handler' ), 10, 3 );
			}

			/**
			 * Add a metabox showing the barcode for the order
			 */
			add_action( 'add_meta_boxes', array( $this, 'add_barcode_metabox' ) );

			/**
			 * If a manual value is entered on the product barcode fields, use it as the current barcode value
			 */
			add_action(
				'save_post_product',
				array(
					$this,
					'save_manual_barcode',
				),
				10,
				3
			);

			add_action( 'wp_ajax_apply_barcode_to_products', array( $this, 'apply_barcode_to_products' ) );

			add_action( 'wp_ajax_nopriv_apply_barcode_to_products', array( $this, 'apply_barcode_to_products' ) );

			add_action( 'wp_ajax_regenerate_barcodes_selector_change', array( $this, 'regenerate_barcodes_selector_change' ) );

			add_action( 'wp_ajax_nopriv_regenerate_barcodes_selector_change', array( $this, 'regenerate_barcodes_selector_change' ) );

			add_action( 'wp_ajax_print_barcodes_selector_change', array( $this, 'print_barcodes_selector_change' ) );

			add_action( 'wp_ajax_nopriv_print_barcodes_selector_change', array( $this, 'print_barcodes_selector_change' ) );

			add_action( 'admin_action_print_all_barcodes', array( $this, 'print_product_barcodes_document' ) );

			add_action( 'admin_action_print_barcodes_by_product', array( $this, 'print_barcodes_by_product_document' ) );

			add_action(
				'wp_ajax_print_barcodes_in_browser',
				array(
					$this,
					'print_barcodes_in_browser',
				)
			);

			add_action(
				'wp_ajax_nopriv_print_barcodes_in_browser',
				array(
					$this,
					'print_barcodes_in_browser',
				)
			);

			add_action(
				'wp_ajax_print_barcodes_by_product_browser',
				array(
					$this,
					'print_barcodes_by_product_browser',
				)
			);

			add_action(
				'wp_ajax_nopriv_print_barcodes_by_product_browser',
				array(
					$this,
					'print_barcodes_by_product_browser',
				)
			);

			/**
			 * Add barcode to variations
			 */
			add_action(
				'woocommerce_product_after_variable_attributes',
				array(
					$this,
					'woocommerce_product_after_variable_attributes',
				),
				10,
				3
			);

			add_action(
				'wp_ajax_create_barcode',
				array(
					$this,
					'create_barcode_callback',
				)
			);

			add_action(
				'wp_ajax_delete_barcode',
				array(
					$this,
					'delete_barcode_callback',
				)
			);
		}

		/**
		 * Create_barcode_callback
		 *
		 * @return void
		 */
		public function create_barcode_callback() {

			$user = wp_get_current_user();

			if ( ! $user instanceof WP_User ) {
				return;
			}

			/** APPLY_FILTERS: yith_barcode_role_check
			*
			* Filter the condition to modify the roles ables of create the barcodes.
			*/
			if ( ! in_array( 'yith_vendor', $user->roles, true ) && ! current_user_can( 'manage_woocommerce' ) && apply_filters( 'yith_barcode_role_check', true ) ) {
				return;
			}

			$result = '';

			if ( isset( $_POST['id'] ) && //phpcs:ignore WordPress.Security.NonceVerification
				isset( $_POST['type'] )//phpcs:ignore WordPress.Security.NonceVerification
			) {

				$id    = sanitize_text_field( wp_unslash( $_POST['id'] ) ); // phpcs:ignore WordPress.Security.NonceVerification
				$type  = sanitize_text_field( wp_unslash( $_POST['type'] ) ); // phpcs:ignore WordPress.Security.NonceVerification
				$value = ( isset( $_POST['value'] ) ) ? sanitize_text_field( wp_unslash( $_POST['value'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification

				if ( 'shop_order' === $type ) {
					$this->create_order_barcode( $id, '', $value );
				} elseif ( ( 'product' === $type ) || ( 'product_variation' === $type ) ) {
					$this->create_product_barcode( $id, '', $value );
				}

				ob_start();
				$post = get_post( $id );
				$this->show_barcode_generation_section( $post );

				$result = ob_get_clean();
			}
			wp_send_json( $result );
		}

		/**
		 * Delete_barcode_callback
		 *
		 * @return void
		 */
		public function delete_barcode_callback() {

			$user = wp_get_current_user();

			if ( ! $user instanceof WP_User ) {
				return;
			}

			/** APPLY_FILTERS: yith_barcode_role_check
			*
			* Filter the condition to modify the roles ables of create the barcodes.
			*/
			if ( ! in_array( 'yith_vendor', $user->roles, true ) && ! current_user_can( 'manage_woocommerce' ) && apply_filters( 'yith_barcode_role_check', true ) ) {
				return;
			}

			$result = '';
			if ( isset( $_POST['id'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification

				$id = sanitize_text_field( wp_unslash( $_POST['id'] ) ); // phpcs:ignore WordPress.Security.NonceVerification

				delete_post_meta( $id, '_ywbc_barcode_protocol' );
				delete_post_meta( $id, '_ywbc_barcode_value' );
				delete_post_meta( $id, '_ywbc_barcode_display_value' );
				delete_post_meta( $id, '_ywbc_barcode_filename' );
				delete_post_meta( $id, 'ywbc_barcode_display_value_custom_field' );
				delete_post_meta( $id, '_ywbc_barcode_image' );

				ob_start();
				$post = get_post( $id );
				$this->show_barcode_generation_section( $post );

				$result = ob_get_clean();
			}
			wp_send_json( $result );
		}

		/**
		 * Woocommerce_product_after_variable_attributes
		 *
		 * @param  mixed $loop loop.
		 * @param  mixed $variation_data variation_data.
		 * @param  mixed $variation variation.
		 * @return void
		 */
		public function woocommerce_product_after_variable_attributes( $loop, $variation_data, $variation ) {
			/* @var WP_Post $variation */ //phpcs:ignore --50%valid code
			$this->show_barcode_generation_section( $variation );
		}

		/**
		 * Save_manual_barcode
		 *
		 * If a manual value is entered on the product barcode fields, use it as the current barcode value
		 *
		 * @param  mixed $post_ID post_ID.
		 * @param  mixed $post post.
		 * @param  mixed $update update.
		 * @return void
		 */
		public function save_manual_barcode( $post_ID, $post, $update ) {
			if ( isset( $_POST['ywbc-value'] ) && ! empty( $_POST['ywbc-value'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification
				// Save the custom value.
				$barcode  = YITH_Barcode::get( $post_ID );
				$protocol = YITH_YWBC()->products_protocol;

				$value      = sanitize_text_field( wp_unslash( $_POST['ywbc-value'] ) ); // phpcs:ignore WordPress.Security.NonceVerification
				$image_path = YITH_YWBC()->get_server_file_path( $post_ID, $protocol, $value );

				$barcode->generate( $protocol, $value, $image_path );
				$barcode->save();
			}
		}

		/**
		 * Enqueue_style
		 *
		 * @param  mixed $hook hook.
		 * @return void
		 */
		public function enqueue_style( $hook ) {
			/**
			 * Add styles
			 */
			$screen_id = get_current_screen()->id;
			if ( ( 'product' === $screen_id ) || ( in_array( $screen_id, array( wc_get_page_screen_id( 'shop-order' ), 'shop_order' ), true ) || ( 'shop_order' === $screen_id ) ) || ( 'yith-plugins_page_yith_woocommerce_barcodes_panel' === $screen_id ) ) {
				wp_enqueue_style(
					'ywbc-style',
					YITH_YWBC_ASSETS_URL . '/css/ywbc-style.css',
					array(),
					YITH_YWBC_VERSION
				);
			}

			wp_enqueue_style(
				'ywbc-shortcodes-style',
				YITH_YWBC_ASSETS_URL . '/css/ywbc-shortcodes-style.css',
				array(),
				YITH_YWBC_VERSION
			);

		}

		/**
		 * Enqueue scripts and styles for the back-end
		 *
		 * @param string $hook hook.
		 */
		public function enqueue_scripts( $hook ) {

			$screen_id = get_current_screen()->id;

			if ( 'edit-product' === $screen_id ) {
				wp_register_script(
					'ywbc-bulk-actions',
					YITH_YWBC_SCRIPTS_URL . yit_load_js_file( 'ywbc-bulk-actions.js' ),
					array(
						'jquery',
					),
					YITH_YWBC_VERSION,
					true
				);

				wp_localize_script(
					'ywbc-bulk-actions',
					'ywbc_bk_data',
					array(
						'action_options' => '<option value="ywbc-generate">' . esc_html__( 'Generate barcode', 'yith-woocommerce-barcodes' ) . '</option>',
					)
				);

				wp_enqueue_script( 'ywbc-bulk-actions' );
			}

			if ( ( isset( $_GET['page'] ) && 'yith_woocommerce_barcodes_panel' === $_GET['page'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification

				wp_register_script(
					'ywbc-ajax-apply-barcode',
					YITH_YWBC_SCRIPTS_URL . yit_load_js_file( 'ywbc-ajax-apply-barcode.js' ),
					array(
						'jquery',
						'jquery-ui-progressbar',
					),
					YITH_YWBC_VERSION,
					true
				);

				$link_to_protocols = '<a href="https://docs.yithemes.com/yith-woocommerce-barcodes/how-to-use-the-plugin/barcode-protocols/" target="_blank">' . esc_html__( 'here', 'yith-woocommerce-barcodes' ) . '</a>';

				$ywbc_params = array(
					'ajax_url' => admin_url( 'admin-ajax.php', is_ssl() ? 'https' : 'http' ),
					'messages' => array(
						'complete_task'         => esc_html__( 'Barcodes applied successfully', 'yith-woocommerce-barcodes' ),
						'error_task'            => esc_html__( 'It is not possible to complete the task', 'yith-woocommerce-barcodes' ),
						'completed_with_errors' => esc_html__( 'It is not possible to complete the task due to errors in the barcode generation.', 'yith-woocommerce-barcodes' ),
						// translators: %s is a placeholder for 'here'.
						'link_to_protocols'     => sprintf( esc_html__( 'Check more info %s about the barcode protocols and the allowed values.', 'yith-woocommerce-barcodes' ), $link_to_protocols ),

					),
				);

				wp_localize_script( 'ywbc-ajax-apply-barcode', 'ywbc_params', $ywbc_params );
				wp_enqueue_script( 'ywbc-ajax-apply-barcode' );
			}

			wp_register_script(
				'ywbc-backend',
				YITH_YWBC_SCRIPTS_URL . yit_load_js_file( 'ywbc-backend.js' ),
				array(
					'jquery',
				),
				YITH_YWBC_VERSION,
				true
			);

			/** APPLY_FILTERS: yith_ywbc_loader
			*
			* Filter the default path to the loading gif.
			*/
			wp_localize_script(
				'ywbc-backend',
				'ywbc_data',
				array(
					'loader'   => apply_filters( 'yith_ywbc_loader', YITH_YWBC_ASSETS_URL . '/images/loading.gif' ),
					'ajax_url' => admin_url( 'admin-ajax.php' ),
				)
			);

			wp_enqueue_script( 'ywbc-backend' );

		}

		/**
		 * On_new_order
		 * Create barcode for new orders if needed
		 *
		 * @param  int $order_id order_id.
		 * @return void
		 */
		public function on_new_order( $order_id ) {

			// Check if barcode are enabled for orders.
			if ( ! YITH_YWBC()->enable_on_orders ) {
				return;
			}

			// Check if barcode should be create automatically.
			if ( ! YITH_YWBC()->create_on_orders ) {
				return;
			}

			/** APPLY_FILTERS: ywbc_before_create_order_barcode
			*
			* Filter the condition create or not the order barcode on new order.
			*
			* @param $order_id ID of the order.
			*/
			if ( apply_filters( 'ywbc_before_create_order_barcode', true, $order_id ) ) {
				$this->create_order_barcode( $order_id );
			}

		}

		/**
		 * Create the barcode values for the order
		 *
		 * @param int    $order_id order_id.
		 * @param string $protocol protocol.
		 * @param string $value value.
		 */
		public function create_order_barcode( $order_id, $protocol = '', $value = '' ) {
			$protocol = $protocol ? $protocol : YITH_YWBC()->orders_protocol;

			$order = wc_get_order( $order_id );

			$value_option = get_option( 'ywbc_order_barcode_type', 'id' );

			if ( 'number' === $value_option ) {
				$value_type = $order->get_order_number();
			} elseif ( 'custom_field' === $value_option ) {

				$custom_field = get_option( 'ywbc_order_barcode_type_custom_field' );

				if ( 'order_url' === $custom_field && 'QRcode' === $protocol ) {
					$my_account_url = get_permalink( get_option( 'woocommerce_myaccount_page_id' ) );
					$value_type     = $my_account_url . 'view-order/' . $order_id;
				} else {
					$value_type = $order->get_meta( $custom_field );

				}
			} else {
				$value_type = $order_id;
			}

			$the_value = $value ? $value : $value_type;

			/** APPLY_FILTERS: yith_barcode_new_order_value
			*
			* Filter the value of the barcode image when creating the order barcode.
			*
			* @param $the_value Formatted value.
			* @param $order_id  ID of the order.
			* @param $protocol  Current protocol.
			* @param $value     Current value.
			*/
			$the_value = apply_filters( 'yith_barcode_new_order_value', $the_value, $order_id, $protocol, $value );

			$this->generate_barcode_image( $order_id, $protocol, $the_value );
		}

		/**
		 * Generate a new barcode instance
		 *
		 * @param int    $object_id the id of the object(WC_Product or WC_Order) associated to this barcode.
		 * @param string $protocol  the protocol to use.
		 * @param string $value     the value to use as the barcode value.
		 *
		 * @return YITH_Barcode
		 */
		public function generate_barcode_image( $object_id, $protocol, $value ) {
			$barcode = new YITH_Barcode( $object_id );

			$image_path = YITH_YWBC()->get_server_file_path( $object_id, $protocol, $value );

			$res = $barcode->generate( $protocol, $value, $image_path );
			$barcode->save();

			/** DO_ACTION: yith_ywbc_after_generate_barcode
			*
			* Adds an action after generating the barcode image.
			*
			* @param $object_id ID of the barcode.
			* @param $protocol  Protocol.
			* @param $value     Value.
			*/
			do_action( 'yith_ywbc_after_generate_barcode', $object_id, $protocol, $value );

			return $barcode;

		}

		/**
		 * Create barcode for new products if needed
		 *
		 * @param string  $new_status new_status.
		 * @param string  $old_status old_status.
		 * @param WP_Post $post post.
		 */
		public function on_new_product( $new_status, $old_status, $post ) {

			/** APPLY_FILTERS: yith_barcodes_post_type_allowed
			*
			* Filter the default product types allowed to create barcodes on.
			*
			* @param array 'products' by defaul (simple products).
			* @param $post  Post object.
			*/
			$post_type_allowed = apply_filters( 'yith_barcodes_post_type_allowed', array( 'product', 'product_variation' ), $post );
			if ( ! in_array( $post->post_type, $post_type_allowed, true ) ) {
				return;
			}

			// Check if barcode are enabled for products.
			if ( ! YITH_YWBC()->enable_on_products ) {
				return;
			}

			// Check if barcode should be create automatically.
			if ( ! YITH_YWBC()->create_on_products ) {
				return;
			}

			// Work only on published posts.
			if ( 'new' !== $old_status ) {
				return;
			}

			$this->create_product_barcode( $post->ID );
		}

		/**
		 * Create the barcode values for the order
		 *
		 * @param int    $product_id product_id.
		 * @param string $protocol protocol.
		 * @param string $value value.
		 *
		 * @return YITH_Barcode
		 */
		public function create_product_barcode( $product_id, $protocol = '', $value = '' ) {

			$protocol = $protocol ? $protocol : YITH_YWBC()->products_protocol;

			$product = wc_get_product( $product_id );

			$value_option = get_option( 'ywbc_product_barcode_type', 'id' );

			if ( 'sku' === $value_option ) {
				$value_type = $product->get_sku();
			} elseif ( 'custom_field' === $value_option ) {
				$custom_field = get_option( 'ywbc_product_barcode_type_custom_field' );
				$value_type   = get_post_meta( $product_id, $custom_field, true );
			} elseif ( 'product_url' === $value_option && 'QRcode' === $protocol ) {
				$value_type = get_permalink( $product_id );
			} else {
				$value_type = $product_id;
			}

			$the_value = $value ? $value : $value_type;

			/** APPLY_FILTERS: yith_barcode_new_product_protocol
			*
			* Filter the protocol when creating the products barcodes.
			*
			* @param $protocol   Current protocol.
			* @param $the_value  Current value.
			*/
			$protocol = apply_filters( 'yith_barcode_new_product_protocol', $protocol, $the_value );

			/** APPLY_FILTERS: yith_barcode_new_product_value
			*
			* Filter the value when creating the products barcodes.
			*
			* @param $the_value  Default the_value.
			* @param $product_id ID of the product.
			* @param $protocol   Current protocol.
			* @param $value      Current value.
			*/
			$the_value = apply_filters( 'yith_barcode_new_product_value', $the_value, $product_id, $protocol, $value );

			return $this->generate_barcode_image( $product_id, $protocol, $the_value );
		}

		/**
		 * Manage a request from product bulk actions
		 */
		public function generate_bulk_action() {

			global $typenow;
			$post_type = $typenow;
			$sendback  = admin_url( "edit.php?post_type=$post_type" );

			$wp_list_table = _get_list_table( 'WP_Posts_List_Table' );
			$action        = $wp_list_table->current_action();

			if ( 'ywbc-generate' === $action ) {
				$post_ids = isset( $_GET['post'] ) ? array_map( 'absint', $_GET['post'] ) : '';

				check_admin_referer( 'bulk-posts' );

				foreach ( $post_ids as $post_id ) {

					$product = wc_get_product( $post_id );

					if ( is_object( $product ) && $product->get_type() === 'variable' ) {

						foreach ( $product->get_available_variations() as $variation ) {

							$this->create_product_barcode( $variation['variation_id'] );
						}
					} else {
						$this->create_product_barcode( $post_id );
					}
				}

				wp_safe_redirect( $sendback );

				exit();
			}

		}

		/**
		 * Generate_orders_bulk_action
		 *
		 * Create new bulk action in the orders page.
		 *
		 * @param  mixed $actions actions.
		 * @return actions
		 */
		public function generate_orders_bulk_action( $actions ) {

			$actions['order_barcode'] = esc_html__( 'Generate barcodes', 'yith-woocommerce-barcodes' );

			return $actions;
		}

		/**
		 * Orders_bulk_actions_handler
		 *
		 * Manage bulk actions of the orders page.
		 *
		 * @param  mixed $redirect_to redirect_to.
		 * @param  mixed $action action.
		 * @param  mixed $post_ids post_ids.
		 * @return redirect_to
		 */
		public function orders_bulk_actions_handler( $redirect_to, $action, $post_ids ) {

			if ( 'order_barcode' === $action ) {
				if ( count( $post_ids ) > 0 ) {
					foreach ( $post_ids as $post_id ) {
						YITH_WooCommerce_Barcodes::get_instance()->create_order_barcode( $post_id );
					}
				}
			}

			return $redirect_to;
		}

		/**
		 * Query_barcode_join
		 *
		 * Set the join part of the query used for filtering products
		 *
		 * @param  string   $join join.
		 * @param  WP_Query $par2 par2.
		 * @return string
		 */
		public function query_barcode_join( $join, $par2 ) {

			if ( empty( $_GET['s'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification
				return $join;
			}

			// Check for necessary arguments.
			if ( ! isset( $par2 ) || ! isset( $par2->query['post_type'] ) ) {
				return $join;
			}

			// Do something only for products and orders.
			if ( ( 'product' !== $par2->query['post_type'] ) && ( 'shop_order' !== $par2->query['post_type'] ) ) {
				return $join;
			}

			global $wpdb;

			$join .= sprintf( " LEFT JOIN {$wpdb->postmeta} ps_meta ON {$wpdb->posts}.ID = ps_meta.post_id and ps_meta.meta_key = '_ywbc_barcode_display_value'" );

			return $join;
		}

		/**
		 * Search_by_order_fields
		 *
		 * @param  mixed $fields fields.
		 * @return fields
		 */
		public function search_by_order_fields( $fields ) {
			$fields[] = YITH_Barcode::YITH_YWBC_META_KEY_BARCODE_DISPLAY_VALUE;

			return $fields;
		}

		/**
		 * Set the where part of the query used for filtering products
		 *
		 * @param string   $where where.
		 * @param WP_Query $par2 par2.
		 *
		 * @return string
		 */
		public function query_barcode_where( $where, $par2 ) {

			if ( empty( $_GET['s'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification
				return $where;
			}

			$search_value = sanitize_text_field( wp_unslash( $_GET['s'] ) ); // phpcs:ignore WordPress.Security.NonceVerification

			// check for necessary arguments.
			if ( ! isset( $par2 ) || ! isset( $par2->query['post_type'] ) ) {
				return $where;
			}

			// Do something only for products and orders.
			if ( ( 'product' !== $par2->query['post_type'] ) && ( 'shop_order' !== $par2->query['post_type'] ) ) {
				return $where;
			}

			global $wpdb;

			$meta = $wpdb->get_results( $wpdb->prepare( 'SELECT * FROM ' . $wpdb->postmeta . ' WHERE meta_key=%s AND meta_value=%s', '_ywbc_barcode_display_value', $_GET['s'] ) ); // phpcs:ignore --Discourage Database call.

			$product = isset( $meta[0]->post_id ) ? wc_get_product( $meta[0]->post_id ) : '';

			if ( $product && $product->get_type() === 'variation' ) {

				$product_parent = wc_get_product( $product->get_parent_id() );

				$search_value = $product_parent->get_id();
			}

			if ( is_object( $product ) && 'product' === $par2->query['post_type'] ) {
				$where .= sprintf( " or (ps_meta.meta_value like '%%%s%%') ", $search_value );

			}

			return $where;
		}
		/**
		 * Add_barcode_metabox
		 *
		 * Show the order metabox
		 *
		 * @param string $post_type Current post type.
		 *
		 * @return void
		 */
		public function add_barcode_metabox( $post_type ) {

			if ( YITH_YWBC()->show_on_order_page && in_array( $post_type, array( wc_get_page_screen_id( 'shop-order' ), 'shop_order' ), true ) ) {

				// Add metabox on order page.
				add_meta_box(
					'ywbc_barcode',
					esc_html__( 'YITH Barcodes', 'yith-woocommerce-barcodes' ),
					array(
						$this,
						'show_barcode_generation_section',
					),
					$post_type,
					'side',
					'high'
				);
			}

			if ( YITH_YWBC()->enable_on_products ) {
				// Add metabox on order page.
				add_meta_box(
					'ywbc_barcode',
					esc_html__( 'YITH Barcodes', 'yith-woocommerce-barcodes' ),
					array(
						$this,
						'show_barcode_generation_section',
					),
					'product',
					'side',
					'high'
				);
			}
		}

		/**
		 * Display the barcode metabox
		 *
		 * @param WP_Post $post post.
		 *
		 * @since  1.0.0
		 */
		public function show_barcode_generation_section( $post ) {
			$post_type = '';

			if ( $post instanceof WC_Order || is_object( $post ) && 'shop_order' === $post->post_type ) {
				$post_type = 'shop_order';
			} elseif ( $post instanceof WC_Product || is_object( $post ) && 'product' === $post->post_type ) {
				$post_type = 'product';
			} elseif ( $post instanceof WC_Product_Varible || is_object( $post ) && 'product_variation' === $post->post_type ) {
				$post_type = 'product_variation';
			}

			if ( ( 'shop_order' === $post_type ) || ( 'product' === $post_type ) || ( 'product_variation' === $post_type ) ) {

				$post_id = $post instanceof WP_Post ? $post->ID : $post->get_id();

				?>
				<div class="ywbc-barcode-generation">
					<?php
					YITH_YWBC()->show_barcode( $post_id );

					/** APPLY_FILTERS: ywbc_show_manual_barcode_section_in_product
					*
					* Filter the condition to show or hide the barcode section in the edit product page.
					*/
					if ( get_option( 'ywbc_product_manual_barcode_product', 'no' ) === 'yes' && 'product' === $post_type || 'product_variation' === $post_type && apply_filters( 'ywbc_show_manual_barcode_section_in_product', true ) ) {

						$this->show_generate_barcode_button( $post_type, $post_id );
					} elseif ( get_option( 'ywbc_enable_on_orders', 'yes' ) === 'yes' && 'shop_order' === $post_type ) {
						$this->show_generate_barcode_button( $post_type, $post_id );
					}

					?>
				</div>
				<?php
			}
		}


		/**
		 * Show a button that let the admin to generate a new barcode for the order
		 *
		 * @param string $type the type of object for which the action generate is intended for.
		 * @param int    $obj_id obj_id.
		 */
		public function show_generate_barcode_button( $type, $obj_id ) {
			?>
			<div class="ywbc-generate-barcode">
				<label for="ywbc-value"><?php esc_html_e( 'Code', 'yith-woocommerce-barcodes' ); ?></label>
				<input type="text" name="ywbc-value" class="ywbc-value-field"/>
				<div>
					<span style="font-size: smaller"><?php esc_html_e( 'Enter the code or leave empty for automatic code', 'yith-woocommerce-barcodes' ); ?></span>
					<br>
					<button class="button button-primary ywbc-generate"
							data-id="<?php echo esc_attr( $obj_id ); ?>"
							data-type="<?php echo esc_attr( $type ); ?>"><?php esc_html_e( 'Generate', 'yith-woocommerce-barcodes' ); ?></button>

					<button class="button button-primary ywbc-delete-barcode" style="margin-top: 10px;"
							data-id="<?php echo esc_attr( $obj_id ); ?>"><?php esc_html_e( 'Delete', 'yith-woocommerce-barcodes' ); ?></button>
				</div>
			</div>
			<?php
		}


		/**
		 * Apply_barcode_to_products
		 *
		 * @since 1.0.2
		 * @return void
		 */
		public function apply_barcode_to_products() {

			$item_id       = isset( $_POST['ywbc_item_id'] ) ? sanitize_text_field( wp_unslash( $_POST['ywbc_item_id'] ) ) : false; // phpcs:ignore WordPress.Security.NonceVerification
			$result        = 'error_on_create_barcode';
			$error_message = '';

			if ( $item_id ) {

				$barcode      = '';
				$protocol     = get_option( 'product_barcode_or_qr', 'barcode' ) === 'barcode' ? get_option( 'ywbc_products_protocol', 'EAN13' ) : 'QRcode';
				$value_option = get_option( 'ywbc_product_barcode_type', 'id' );

				if ( 'sku' === $value_option ) {

					$product = wc_get_product( $item_id );
					$value   = $product->get_sku();

				} elseif ( 'custom_field' === $value_option ) {

					$custom_field = get_option( 'ywbc_product_barcode_type_custom_field' );
					$value        = get_post_meta( $item_id, $custom_field, true );

				} elseif ( 'product_url' === $value_option && 'QRcode' === $protocol ) {

					$value = get_permalink( $item_id );

				} else {

					$value = $item_id;
				}

				try {

					/** APPLY_FILTERS: ywbc_skip_value_check_on_barcode_generation
					*
					* Filter the condition to skip the check in the barcode generation.
					*/
					if ( ! empty( $value ) || apply_filters( 'ywbc_skip_value_check_on_barcode_generation', false ) ) {
						$barcode = $this->create_product_barcode( $item_id );
					} else {
						throw new Exception();
					}
				} catch ( Exception $e ) {
					$result = 'barcode_wrong_format';
					$value  = ! empty( $value ) ? $value : esc_html__( 'Empty value!', 'yith-woocommerce-barcodes' );
					// translators: %s are placeholders for the protocol, the item ID and the value.
					$error_message = $e->getMessage() . ' ' . sprintf( esc_html_x( 'on the %s protocol. Please, select another compatible protocol or check the value for the product with ID: %s. The actual value is: %s', 'error message when a barcode cannot be generated using the tool, the string starts with the PHP error generated', 'yith-woocommerce-barcodes' ), $protocol,  $item_id, $value );
				}

				if ( is_object( $barcode ) && $barcode->object_id ) {
					$result = 'barcode_created';
				}
			}
			wp_send_json(
				array(
					'result'        => $result,
					'error_message' => $error_message,
				)
			);
		}

		/**
		 * Regenerate_barcodes_selector_change
		 *
		 * @return void
		 */
		public function regenerate_barcodes_selector_change() {
			$mode = ( isset( $_POST['mode'] ) ) ? sanitize_text_field( wp_unslash( $_POST['mode'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification
			if ( 'generate' === $mode ) {
				update_option( 'ywbc_tool_regenerate_barcodes', 'generate' );
			} else {
				update_option( 'ywbc_tool_regenerate_barcodes', 'regenerate' );
			}
		}

		/**
		 * Print_barcodes_selector_change
		 *
		 * @return void
		 */
		public function print_barcodes_selector_change() {
			$mode = ( isset( $_POST['mode'] ) ) ? sanitize_text_field( wp_unslash( $_POST['mode'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification

			if ( 'include_variations' === $mode ) {
				update_option( 'ywbc_enable_print_barcodes_variations', 'include_variations' );
			} elseif ( 'all_products' === $mode ) {
				update_option( 'ywbc_enable_print_barcodes_variations', 'all_products' );
			} else {
				update_option( 'ywbc_enable_print_barcodes_variations', 'all_stock' );
			}
		}

		/**
		 * Print_product_barcodes_document
		 *
		 * Print all products barcodes
		 *
		 * @since 1.0.2
		 */
		public function print_product_barcodes_document() {

			$selected_option = get_option( 'ywbc_enable_print_barcodes_variations', 'all_products' );

			$post_types = 'include_variations' === $selected_option || 'all_stock' === $selected_option ? array( 'product', 'product_variation' ) : 'product';

			$print_all_stock = 'all_stock' === $selected_option ? true : false;

			/** APPLY_FILTERS: yith_barcode_query_args_get_products
			*
			* Filter the query when creating the document to be printed.
			*
			* @param array $post_type, $post_status, $post_per_page, $fields, $meta_query.
			*/
			$query_args = apply_filters(
				'yith_barcode_query_args_get_products',
				array(

					/** APPLY_FILTERS: yith_barcode_query_post_type
					*
					* Filter the post types on the query when creating the document to be printed.
					*
					* @param array $post_types Default post types to be retrieved.
					*/
					'post_type'      => apply_filters( 'yith_barcode_query_post_type', $post_types ),
					'post_status'    => 'publish',
					'posts_per_page' => -1,
					'fields'         => 'ids',
					'meta_query'     => array( // phpcs:ignore slow query ok.
						array(
							'key'     => YITH_Barcode::YITH_YWBC_META_KEY_BARCODE_DISPLAY_VALUE,
							'compare' => 'EXISTS',
						),
					),
				)
			);

			$item_ids = get_posts( $query_args );

			require_once YITH_YWBC_DIR . 'includes/vendor/autoload.php';

			/** APPLY_FILTERS: yith_ywbc_mpdf_args
			*
			* Filter the mpdf arguments when creating the document to be printed.
			*
			* @param array $autoScriptToLang, $autoLangToFont.
			*/
			$mpdf_args = apply_filters(
				'yith_ywbc_mpdf_args',
				array(
					'autoScriptToLang' => true,
					'autoLangToFont'   => true,
				)
			);

			if ( is_array( $mpdf_args ) ) {
				$mpdf = new \Mpdf\Mpdf( $mpdf_args );
			} else {
				$mpdf = new \Mpdf\Mpdf();
			}

			ob_start();

			wc_get_template(
				'/print/print-barcodes-template-style.css',
				null,
				'',
				YITH_YWBC_TEMPLATE_PATH
			);

			$style = ob_get_clean();

			ob_start();

			$args = array(
				'item_ids'        => $item_ids,
				'context'         => 'pdf',
				'print_all_stock' => $print_all_stock,
			);

			wc_get_template( '/print/print-barcodes-template.php', $args, '', YITH_YWBC_TEMPLATE_PATH );

			$html = ob_get_clean();

			/** APPLY_FILTERS: yith_ywbc_before_rendering_barcodes_mpdf_html
			*
			* Filter the HTML before rendering the barcode to be printed.
			*
			* @param $html Default HTML.
			*/
			$html = apply_filters( 'yith_ywbc_before_rendering_barcodes_mpdf_html', $html );

			/** DO_ACTION: yith_ywbc_after_write_mpdf_html_template_pdf
			*
			* Adds an action after write the HTML template PDF.
			*
			* @param $mpdf MPDF instance.
			*/
			do_action( 'yith_ywbc_after_write_mpdf_html_template_pdf', $mpdf );

			$mpdf->WriteHTML( $style, 1 );

			$mpdf->WriteHTML( $html, 2 );

			/** DO_ACTION: yith_ywbc_after_generate_template_pdf
			*
			* Adds an action after write the template PDF.
			*
			* @param $mpdf MPDF instance.
			*/
			do_action( 'yith_ywbc_after_generate_template_pdf', $mpdf );

			$pdf = $mpdf->Output( 'document', 'S' );

			$pdf_filename = get_option( 'ywbc_pdf_file_name', 'yith-barcode-[uniqid]' );

			/** APPLY_FILTERS: yith_ywbc_formatted_pdf_filename
			*
			* Filter the file name of the pdf that's gonna be printed.
			*
			* @param $pdf_filename Default file name.
			*/
			$formatted_pdf_filename = apply_filters(
				'yith_ywbc_formatted_pdf_filename',
				str_replace(
					array( '[uniqid]' ),
					array( uniqid() ),
					$pdf_filename
				),
				$pdf_filename
			);

			/** APPLY_FILTERS: yith_ywbc_pdf_new_file_path
			*
			* Filter the path of the pdf that's gonna be printed.
			*/
			$new_file = apply_filters( 'yith_ywbc_pdf_new_file_path', YITH_YWBC_UPLOAD_DIR . '/' . $formatted_pdf_filename . '.pdf' );

			file_put_contents( $new_file, $pdf );

			header( 'Content-type:  application/pdf' );
			header( 'Content-Disposition: attachment; filename="' . basename( $new_file ) . '"' );

			readfile( $new_file );

			ignore_user_abort( true );

			unlink( $new_file );

			exit;
		}

		/**
		 * Print_barcodes_by_product_document
		 *
		 * @since 2.0.0
		 */
		public function print_barcodes_in_browser() {

			$selected_option = get_option( 'ywbc_enable_print_barcodes_variations', 'all_products' );

			$post_types = 'include_variations' === $selected_option || 'all_stock' === $selected_option ? array( 'product', 'product_variation' ) : 'product';

			$print_all_stock = 'all_stock' === $selected_option ? true : false;

			/** APPLY_FILTERS: yith_barcode_query_args_get_products
			*
			* Filter the query when creating the document to be printed.
			*
			* @param array $post_type, $post_status, $post_per_page, $fields, $meta_query.
			*/
			$query_args = apply_filters(
				'yith_barcode_query_args_get_products',
				array(

					/** APPLY_FILTERS: yith_barcode_query_post_type
					*
					* Filter the post types on the query when creating the document to be printed.
					*
					* @param array $post_types Default post types to be retrieved.
					*/
					'post_type'      => apply_filters( 'yith_barcode_query_post_type', $post_types ),
					'post_status'    => 'publish',
					'posts_per_page' => -1,
					'fields'         => 'ids',
					'meta_query'     => array( // phpcs:ignore slow query ok.
						array(
							'key'     => YITH_Barcode::YITH_YWBC_META_KEY_BARCODE_DISPLAY_VALUE,
							'compare' => 'EXISTS',
						),
					),
				)
			);

			$item_ids = get_posts( $query_args );

			$args = array(
				'item_ids'        => $item_ids,
				'context'         => 'browser',
				'print_all_stock' => $print_all_stock,
			);

			ob_start();
			wc_get_template( '/print/print-barcodes-template.php', $args, '', YITH_YWBC_TEMPLATE_PATH );
			$print_template = ob_get_clean();

			wp_send_json( array( 'result' => $print_template ) );

		}

		/**
		 * Print_barcodes_by_product_document
		 *
		 * @since 2.0.0
		 */
		public function print_barcodes_by_product_browser() {

			$product_id = isset( $_POST['ywbc_product_id'] ) ? sanitize_text_field( wp_unslash( $_POST['ywbc_product_id'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification
			$quantity   = isset( $_POST['ywbc_product_quantity'] ) ? sanitize_text_field( wp_unslash( $_POST['ywbc_product_quantity'] ) ) : 1; // phpcs:ignore WordPress.Security.NonceVerification

			$args = array(
				'product_id' => $product_id,
				'quantity'   => $quantity,
				'context'    => 'browser',
			);

			ob_start();
			wc_get_template( '/print/print-barcodes-by-product-template.php', $args, '', YITH_YWBC_TEMPLATE_PATH );
			$print_template = ob_get_clean();

			wp_send_json( array( 'result' => $print_template ) );

		}

		/**
		 * Print_barcodes_by_product_document
		 *
		 * @since 2.0.0
		 */
		public function print_barcodes_by_product_document() {

			$product_id = isset( $_REQUEST['product_id'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['product_id'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification
			$quantity   = isset( $_REQUEST['product_quantity'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['product_quantity'] ) ) : 1; // phpcs:ignore WordPress.Security.NonceVerification

			if ( ! $product_id ) {
				return;
			}

			require_once YITH_YWBC_DIR . 'includes/vendor/autoload.php';

			/** APPLY_FILTERS: yith_ywbc_mpdf_args
			*
			* Filter the mpdf arguments when creating the document to be printed.
			*
			* @param array $autoScriptToLang, $autoLangToFont.
			*/
			$mpdf_args = apply_filters(
				'yith_ywbc_mpdf_args',
				array(
					'autoScriptToLang' => true,
					'autoLangToFont'   => true,
				)
			);

			if ( is_array( $mpdf_args ) ) {
				$mpdf = new \Mpdf\Mpdf( $mpdf_args );
			} else {
				$mpdf = new \Mpdf\Mpdf();
			}

			ob_start();

			wc_get_template(
				'/print/print-barcodes-template-style.css',
				null,
				'',
				YITH_YWBC_TEMPLATE_PATH
			);

			$style = ob_get_clean();

			ob_start();

			$args = array(
				'product_id' => $product_id,
				'quantity'   => $quantity,
				'context'    => 'pdf',
			);

			wc_get_template( '/print/print-barcodes-by-product-template.php', $args, '', YITH_YWBC_TEMPLATE_PATH );

			$html = ob_get_clean();

			/** APPLY_FILTERS: yith_ywbc_before_rendering_barcodes_mpdf_html
			*
			* Filter the HTML before rendering the barcode to be printed.
			*
			* @param $html Default HTML.
			*/
			$html = apply_filters( 'yith_ywbc_before_rendering_barcodes_mpdf_html', $html );

			/** DO_ACTION: yith_ywbc_after_write_mpdf_html_template_pdf
			*
			* Adds an action after write the HTML template PDF.
			*
			* @param $mpdf MPDF instance.
			*/
			do_action( 'yith_ywbc_after_write_mpdf_html_template_pdf', $mpdf );

			$mpdf->WriteHTML( $style, 1 );

			$mpdf->WriteHTML( $html, 2 );

			/** DO_ACTION: yith_ywbc_after_generate_template_pdf
			*
			* Adds an action after write the template PDF.
			*
			* @param $mpdf MPDF instance.
			*/
			do_action( 'yith_ywbc_after_generate_template_pdf', $mpdf );

			$pdf = $mpdf->Output( 'document', 'S' );

			$pdf_filename = get_option( 'ywbc_pdf_file_name', 'yith-barcode-[uniqid]' );

			/** APPLY_FILTERS: yith_ywbc_formatted_pdf_filename
			*
			* Filter the file name of the pdf that's gonna be printed.
			*
			* @param $pdf_filename Default file name.
			*/
			$formatted_pdf_filename = apply_filters(
				'yith_ywbc_formatted_pdf_filename',
				str_replace(
					array( '[uniqid]' ),
					array( uniqid() ),
					$pdf_filename
				),
				$pdf_filename
			);

			/** APPLY_FILTERS: yith_ywbc_pdf_new_file_path
			*
			* Filter the path of the pdf that's gonna be printed.
			*/
			$new_file = apply_filters( 'yith_ywbc_pdf_new_file_path', YITH_YWBC_UPLOAD_DIR . '/' . $formatted_pdf_filename . '.pdf' );

			file_put_contents( $new_file, $pdf );

			header( 'Content-type:  application/pdf' );
			header( 'Content-Disposition: attachment; filename="' . basename( $new_file ) . '"' );

			readfile( $new_file );

			ignore_user_abort( true );

			unlink( $new_file );

			exit;
		}

		/**
		 * Load_gutenberg_compatibility
		 *
		 * @return void
		 */
		public function load_gutenberg_compatibility() {
			$blocks = array(
				'yith-product-barcode' => array(
					'style'          => 'ywbc-style',
					'script'         => 'ywbc-frontend',
					'title'          => _x( 'Product Barcode', '[gutenberg]: block name', 'yith-woocommerce-barcodes' ),
					'description'    => _x( 'With this block you can search a product by barcode and manage the stock', '[gutenberg]: block description', 'yith-woocommerce-barcodes' ),
					'shortcode_name' => 'yith_product_barcode',
					'callback'       => 'yith_product_barcode',
					'do_shortcode'   => true,
					'keywords'       => array(
						_x( 'Product', '[gutenberg]: keywords', 'yith-woocommerce-barcodes' ),
						_x( 'Barcode', '[gutenberg]: keywords', 'yith-woocommerce-barcodes' ),
						_x( 'Stock', '[gutenberg]: keywords', 'yith-woocommerce-barcodes' ),
						_x( 'Search', '[gutenberg]: keywords', 'yith-woocommerce-barcodes' ),
					),
				),
				'yith-order-barcode'   => array(
					'style'          => 'ywbc-style',
					'title'          => _x( 'Order Barcode', '[gutenberg]: block name', 'yith-woocommerce-barcodes' ),
					'description'    => _x( 'With this block you can search an order by barcode and manage its status', '[gutenberg]: block description', 'yith-woocommerce-barcodes' ),
					'shortcode_name' => 'yith_order_barcode',
					'callback'       => 'yith_order_barcode',
					'do_shortcode'   => true,
					'keywords'       => array(
						_x( 'Order', '[gutenberg]: keywords', 'yith-woocommerce-barcodes' ),
						_x( 'Barcode', '[gutenberg]: keywords', 'yith-woocommerce-barcodes' ),
						_x( 'Status', '[gutenberg]: keywords', 'yith-woocommerce-barcodes' ),
						_x( 'Search', '[gutenberg]: keywords', 'yith-woocommerce-barcodes' ),
					),
				),
				'yith-render-barcode'  => array(
					'style'          => 'ywbc-style',
					'title'          => _x( 'Render Barcode', '[gutenberg]: block name', 'yith-woocommerce-barcodes' ),
					'description'    => _x( 'With this block you can show a barcode by product or order ID', '[gutenberg]: block description', 'yith-woocommerce-barcodes' ),
					'shortcode_name' => 'yith_render_barcode',
					'callback'       => 'yith_render_barcode',
					'do_shortcode'   => true,
					'keywords'       => array(
						_x( 'Order', '[gutenberg]: keywords', 'yith-woocommerce-barcodes' ),
						_x( 'Barcode', '[gutenberg]: keywords', 'yith-woocommerce-barcodes' ),
						_x( 'Status', '[gutenberg]: keywords', 'yith-woocommerce-barcodes' ),
						_x( 'Render', '[gutenberg]: keywords', 'yith-woocommerce-barcodes' ),
					),
					'attributes'     => array(
						'id'            => array(
							'type'    => 'text',
							'label'   => _x( 'ID', '[gutenberg]: Option title', 'yith-woocommerce-barcodes' ),
							'default' => 0,
						),
						'hide_if_empty' => array(
							'type'    => 'checkbox',
							'label'   => _x( 'Hide if empty', '[gutenberg]: Option title', 'yith-woocommerce-barcodes' ),
							'default' => true,
							'helps'   => array(
								'checked'   => _x( 'Hide if empty', '[gutenberg]: Help text', 'yith-woocommerce-barcodes' ),
								'unchecked' => _x( 'Show if empty', '[gutenberg]: Help text', 'yith-woocommerce-barcodes' ),
							),
						),
						'value'         => array(
							'type'    => 'text',
							'label'   => _x( 'Value', '[gutenberg]: Option title', 'yith-woocommerce-barcodes' ),
							'default' => '',
						),
						'protocol'      => array(
							'type'    => 'radio',
							'label'   => _x( 'Select protocol to use', '[gutenberg]: block description', 'yith-woocommerce-barcodes' ),
							'options' => array(
								'EAN13'   => _x( 'EAN-13', '[gutenberg]: inspector description', 'yith-woocommerce-barcodes' ),
								'EAN8'    => _x( 'EAN-8', '[gutenberg]: inspector description', 'yith-woocommerce-barcodes' ),
								'STD25'   => _x( 'STD 25', '[gutenberg]: inspector description', 'yith-woocommerce-barcodes' ),
								'INT25'   => _x( 'INT 25', '[gutenberg]: inspector description', 'yith-woocommerce-barcodes' ),
								'CODE39'  => _x( 'CODE 39', '[gutenberg]: inspector description', 'yith-woocommerce-barcodes' ),
								'code93'  => _x( 'CODE 93', '[gutenberg]: inspector description', 'yith-woocommerce-barcodes' ),
								'code128' => _x( 'CODE 128', '[gutenberg]: inspector description', 'yith-woocommerce-barcodes' ),
								'Codabar' => _x( 'Codabar', '[gutenberg]: inspector description', 'yith-woocommerce-barcodes' ),
								'QRcode'  => _x( 'QR code', '[gutenberg]: inspector description', 'yith-woocommerce-barcodes' ),
							),
							'default' => 'EAN13',
						),
						'inline_css'    => array(
							'type'    => 'text',
							'label'   => _x( 'Inline CSS', '[gutenberg]: Option title', 'yith-woocommerce-barcodes' ),
							'default' => '',
						),
					),
				),
			);

			yith_plugin_fw_gutenberg_add_blocks( $blocks );
		}

	}
}

YITH_YWBC_Backend::get_instance();
