<?php
/**
 * This file belongs to the YIT Framework.
 *
 * This source file is subject to the GNU GENERAL PUBLIC LICENSE (GPL 3.0)
 * that is bundled with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * http://www.gnu.org/licenses/gpl-3.0.txt
 *
 * @package YITH\PreOrder\Includes
 * @author YITH <plugins@yithemes.com>
 */

if ( ! defined( 'YITH_WCPO_VERSION' ) ) {
	exit( 'Direct access forbidden.' );
}

if ( ! class_exists( 'YITH_Pre_Order_Edit_Product_Page_Premium' ) ) {
	/**
	 * Class YITH_Pre_Order_Edit_Product_Page_Premium
	 */
	class YITH_Pre_Order_Edit_Product_Page_Premium extends YITH_Pre_Order_Edit_Product_Page {

		/**
		 * Returns single instance of the class
		 *
		 * @return YITH_Pre_Order_Edit_Product_Page_Premium
		 * @since 1.0.0
		 */
		public static function get_instance() {
			if ( is_null( self::$instance ) ) {
				self::$instance = new self();
			}

			return self::$instance;
		}

        /**
         * Enqueue the scripts for Edit product page.
         */
        public function enqueue_scripts() {
            $current_screen      = get_current_screen();
            $integrated_gateways = ywpo_exist_integrated_gateways();

            wp_enqueue_script( 'yith-plugin-fw-fields' );
            wp_enqueue_style( 'yith-plugin-fw-fields' );

            if ( 'product' === $current_screen->id ) {
                wp_enqueue_style( 'ywpo-edit-product', YITH_WCPO_ASSETS_URL . 'css/ywpo-edit-product.css', array(), YITH_WCPO_VERSION );
            }

            if ( isset( $_GET['page'] ) && 'yith_wcpo_panel' === $_GET['page'] ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended
                wp_enqueue_style( 'ywpo-pre-order-list-table', YITH_WCPO_ASSETS_URL . 'css/ywpo-pre-order-list-table.css', array(), YITH_WCPO_VERSION );
            }

            if ( 'product' === $current_screen->id || 'edit-shop_order' === $current_screen->id || 'edit-product' === $current_screen->id ) {
                wp_enqueue_script( 'yith-wcpo-edit-product-page', YITH_WCPO_ASSETS_JS_URL . yit_load_js_file( 'edit-product-page.js' ), array( 'jquery' ), YITH_WCPO_VERSION, 'true' );
                wp_localize_script( 'yith-wcpo-edit-product-page', 'ywpo_options', array( 'integrated_gateways' => $integrated_gateways ) );
                wp_enqueue_style( 'ywpo-admin', YITH_WCPO_ASSETS_URL . 'css/ywpo-admin.css', array(), YITH_WCPO_VERSION );
                if ( 'yes' === get_option( 'yith_wcpo_enable_pre_order_auto_outofstock_notification' ) ) {
                    wp_add_inline_script( 'yith-wcpo-edit-product-page', "jQuery( '.yith_ywpo_automatic_pre_order_status #_ywpo_preorder' ).prop( 'disabled', true );" );
                }

                wp_enqueue_script( 'jquery-ui-datetimepicker', YITH_WCPO_ASSETS_JS_URL . yit_load_js_file( 'timepicker.js' ), array( 'jquery' ), YITH_WCPO_VERSION, 'true' );
                wp_enqueue_style( 'jquery-ui-datetimepicker-style', YITH_WCPO_ASSETS_URL . 'css/timepicker.css', array(), YITH_WCPO_VERSION );
            }
        }

		/**
		 * Adds a Pre Order settings tab in the product data box
		 *
		 * @param array $product_data_tabs Edit product page tabs.
		 * @return   array product_data_tabs
		 */
		public function pre_order_tab( $product_data_tabs ) {
			$preorder_tab = array(
				'preorder' => array(
					'label'  => esc_html__( 'Pre-Order', 'yith-pre-order-for-woocommerce' ),
					'target' => 'preorder_product_data',
					'class'  => array( 'show_if_simple' ),
				),
			);

			return array_merge( $product_data_tabs, $preorder_tab );
		}

		/**
		 * Call the template for the settings for the "Pre-Order" tab (Edit product page).
		 */
		public function pre_order_tab_content() {
			global $thepostid;
			$args = self::get_pre_order_post_meta( $thepostid );

			wc_get_template( 'admin/ywpo-edit-product-pre-order-tab.php', $args, '', YITH_WCPO_TEMPLATE_PATH . '/' );
		}

		/**
		 * Call the template for the settings for the "Pre-order options" section inside the "Variations" tab (Edit product page).
		 *
		 * @param int     $loop           Position in the loop.
		 * @param array   $variation_data Variation data.
		 * @param WP_Post $variation      Post data.
		 */
		public function pre_order_variation_content( $loop, $variation_data, $variation ) {
			$args         = self::get_pre_order_post_meta( $variation->ID );
			$args['loop'] = $loop;

			wc_get_template( 'admin/ywpo-edit-product-pre-order-variation.php', $args, '', YITH_WCPO_TEMPLATE_PATH . '/' );
		}

		/**
		 * Collect the pre-order post meta for the edit product page.
		 *
		 * @param string|int $id The product's ID.
		 * @return array
		 */
		private static function get_pre_order_post_meta( $id ) {
			$product   = wc_get_product( $id );

			$pre_order_status = 'yes' === YITH_Pre_Order_Utils::get_pre_order_status( $product ) ? 'yes' : 'no';

			$pre_order_mode          = YITH_Pre_Order_Utils::get_pre_order_mode( $product );
			$pre_order_mode          = ! $pre_order_mode || ! $pre_order_mode && 'yes' === get_option( 'yith_wcpo_enable_pre_order_auto_outofstock_notification' ) ? 'manual' : $pre_order_mode;
			$override_pre_order_mode = get_post_meta( $id, '_ywpo_override_pre_order_mode', true ) ? get_post_meta( $id, '_ywpo_override_pre_order_mode', true ) : 'auto';

			$start_mode       = YITH_Pre_Order_Utils::get_start_mode( $product ) ?: 'now';
			$start_date       = YITH_Pre_Order_Utils::get_start_date( $product );
			$start_date_label = YITH_Pre_Order_Utils::get_start_date_label( $product );

			// Backward compatibility.
			if ( ! metadata_exists( 'post', $id, '_ywpo_availability_date_mode' ) && metadata_exists( 'post', $id, '_ywpo_for_sale_date' ) ) {
				$availability_date_mode = get_post_meta( $id, '_ywpo_for_sale_date', true ) > 0 ? 'date' : 'no_date';
			} else {
				$availability_date_mode = metadata_exists( 'post', $id, '_ywpo_availability_date_mode' ) ? YITH_Pre_Order_Utils::get_availability_date_mode( $product ) : 'no_date';
			}

			$availability_date         = YITH_Pre_Order_Utils::get_for_sale_date( $product );
			$dynamic_availability_date = metadata_exists( 'post', $id, '_ywpo_dynamic_availability_date' ) ? YITH_Pre_Order_Utils::get_dynamic_availability_date( $product ) : '1';

			$preorder_price = metadata_exists( 'post', $id, '_ywpo_preorder_price' ) ? YITH_Pre_Order_Utils::get_pre_order_price( $product ) : 0;

			// Backward compatibility.
			if ( ! metadata_exists( 'post', $id, '_ywpo_price_mode' ) && metadata_exists( 'post', $id, '_ywpo_preorder_price' ) ) {
				$price_mode          = 'default';
				$discount_percentage = '';
				$discount_fixed      = '';
				$increase_percentage = '';
				$increase_fixed      = '';
				if ( metadata_exists( 'post', $id, '_ywpo_price_adjustment' ) && 'manual' === get_post_meta( $id, '_ywpo_price_adjustment', true ) ) {
					$price_mode = 'fixed';
				} elseif (
					metadata_exists( 'post', $id, '_ywpo_price_adjustment' ) &&
					metadata_exists( 'post', $id, '_ywpo_adjustment_type' ) &&
					metadata_exists( 'post', $id, '_ywpo_price_adjustment_amount' )
				) {
					if (
						'discount' === get_post_meta( $id, '_ywpo_price_adjustment', true ) &&
						'percentage' === get_post_meta( $id, '_ywpo_adjustment_type', true )
					) {
						$price_mode          = 'discount_percentage';
						$discount_percentage = get_post_meta( $id, '_ywpo_price_adjustment_amount', true );
					} elseif (
						'discount' === get_post_meta( $id, '_ywpo_price_adjustment', true ) &&
						'fixed' === get_post_meta( $id, '_ywpo_adjustment_type', true )
					) {
						$price_mode     = 'discount_fixed';
						$discount_fixed = get_post_meta( $id, '_ywpo_price_adjustment_amount', true );
					} elseif (
						'mark-up' === get_post_meta( $id, '_ywpo_price_adjustment', true ) &&
						'percentage' === get_post_meta( $id, '_ywpo_adjustment_type', true )
					) {
						$price_mode          = 'increase_percentage';
						$increase_percentage = get_post_meta( $id, '_ywpo_price_adjustment_amount', true );
					} elseif (
						'mark-up' === get_post_meta( $id, '_ywpo_price_adjustment', true ) &&
						'fixed' === get_post_meta( $id, '_ywpo_adjustment_type', true )
					) {
						$price_mode     = 'increase_fixed';
						$increase_fixed = get_post_meta( $id, '_ywpo_price_adjustment_amount', true );
					}
				}
			} else {
				$price_mode          = metadata_exists( 'post', $id, '_ywpo_price_mode' ) ? YITH_Pre_Order_Utils::get_price_mode( $product ) : 'default';
				$discount_percentage = YITH_Pre_Order_Utils::get_discount_percentage( $product );
				$discount_fixed      = YITH_Pre_Order_Utils::get_discount_fixed( $product );
				$increase_percentage = YITH_Pre_Order_Utils::get_increase_percentage( $product );
				$increase_fixed      = YITH_Pre_Order_Utils::get_increase_fixed( $product );
			}

			$max_qty_enabled = metadata_exists( 'post', $id, '_ywpo_max_qty_enabled' ) ? YITH_Pre_Order_Utils::get_max_qty_enabled( $product ) : 'no';
			$max_qty         = metadata_exists( 'post', $id, '_ywpo_max_qty' ) ? YITH_Pre_Order_Utils::get_max_qty( $product ) : 0;

			// Backward compatibility.
			if ( ! metadata_exists( 'post', $id, '_ywpo_override_labels' ) && metadata_exists( 'post', $id, '_ywpo_preorder_label' ) ) {
				$override_labels = 'yes';
			} else {
				$override_labels = YITH_Pre_Order_Utils::get_override_labels( $product ) ?: 'no';
			}

			$pre_order_label         = YITH_Pre_Order_Utils::get_pre_order_label( $product );
			$availability_date_label = YITH_Pre_Order_Utils::get_pre_order_availability_date_label( $product );
			$no_date_label           = metadata_exists( 'post', $id, '_ywpo_preorder_no_date_label' ) ? YITH_Pre_Order_Utils::get_pre_order_no_date_label( $product ) : __( 'Coming soon...', 'yith-pre-order-for-woocommerce' );
			$override_fee            = metadata_exists( 'post', $id, '_ywpo_override_fee' ) ? YITH_Pre_Order_Utils::get_override_fee( $product ) : 'no';
			$fee                     = metadata_exists( 'post', $id, '_ywpo_fee' ) ? YITH_Pre_Order_Utils::get_fee( $product ) : 0;
			$override_charge_type    = metadata_exists( 'post', $id, '_ywpo_override_charge_type' ) ? YITH_Pre_Order_Utils::get_override_charge_type( $product ) : 'no';
			$charge_type             = metadata_exists( 'post', $id, '_ywpo_charge_type' ) ? YITH_Pre_Order_Utils::get_charge_type( $product ) : 'upfront';

			return apply_filters(
				'ywpo_get_pre_order_post_meta',
				array(
					'product'                   => $product,
					'pre_order_status'          => $pre_order_status,
					'pre_order_mode'            => $pre_order_mode,
					'override_pre_order_mode'   => $override_pre_order_mode,
					'start_mode'                => $start_mode,
					'start_date'                => $start_date,
					'start_date_label'          => $start_date_label,
					'availability_date_mode'    => $availability_date_mode,
					'availability_date'         => $availability_date,
					'dynamic_availability_date' => $dynamic_availability_date,
					'price_mode'                => $price_mode,
					'preorder_price'            => $preorder_price,
					'discount_percentage'       => $discount_percentage,
					'discount_fixed'            => $discount_fixed,
					'increase_percentage'       => $increase_percentage,
					'increase_fixed'            => $increase_fixed,
					'max_qty_enabled'           => $max_qty_enabled,
					'max_qty'                   => $max_qty,
					'override_labels'           => $override_labels,
					'pre_order_label'           => $pre_order_label,
					'availability_date_label'   => $availability_date_label,
					'no_date_label'             => $no_date_label,
					'override_fee'              => $override_fee,
					'fee'                       => $fee,
					'override_charge_type'      => $override_charge_type,
					'charge_type'               => $charge_type,
				)
			);
		}

		/**
		 * Update the pre-order post meta.
		 *
		 * @param string|int $post_id The post ID.
		 */
		public function update_product_post_meta( $post_id ) {
			// Check the nonce.
			if ( empty( $_POST['woocommerce_meta_nonce'] ) || ! wp_verify_nonce( wp_unslash( $_POST['woocommerce_meta_nonce'] ), 'woocommerce_save_data' ) ) { // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
				return;
			}

			$product = wc_get_product( $post_id );
			if ( $product->is_type( 'variable' ) ) {
				$children = $product->get_children();
				if ( empty( $children ) ) {
					return;
				}
				$count    = count( $children );
				$enabled  = 0;
				$disabled = 0;
				foreach ( $children as $variation_id ) {
					if ( ! metadata_exists( 'post', $variation_id, '_ywpo_preorder' ) || 'no' === get_post_meta( $variation_id, '_ywpo_preorder', true ) ) {
						$disabled++;
					}
					if ( 'yes' === get_post_meta( $variation_id, '_ywpo_preorder', true ) ) {
						$enabled++;
					}
				}
				if ( $enabled > 0 ) {
					update_post_meta( $post_id, '_ywpo_has_variations', 'yes' );
				} elseif ( metadata_exists( 'post', $post_id, '_ywpo_has_variations' ) && $count === $disabled ) {
					update_post_meta( $post_id, '_ywpo_has_variations', 'no' );
				}
			} else {
				if ( isset( $_POST['product-type'] ) && 'simple' !== $_POST['product-type'] ) {
					return;
				}

				$is_pre_order = ! empty( $_POST['_ywpo_preorder'] ) ? 'yes' : 'no';
				YITH_Pre_Order_Utils::set_pre_order_status( $product, $is_pre_order );

				$pre_order_mode = isset( $_POST['_ywpo_pre_order_mode'] ) ? sanitize_text_field( wp_unslash( $_POST['_ywpo_pre_order_mode'] ) ) : '';
				if ( $pre_order_mode ) {
					YITH_Pre_Order_Utils::set_pre_order_mode( $product, $pre_order_mode );
				}

				$override_pre_order_mode = isset( $_POST['_ywpo_override_pre_order_mode'] ) ? sanitize_text_field( wp_unslash( $_POST['_ywpo_override_pre_order_mode'] ) ) : '';
				if ( $override_pre_order_mode ) {
					YITH_Pre_Order_Utils::set_override_pre_order_mode( $product, $override_pre_order_mode );
				}

				$start_mode = isset( $_POST['_ywpo_start_mode'] ) ? sanitize_text_field( wp_unslash( $_POST['_ywpo_start_mode'] ) ) : '';
				YITH_Pre_Order_Utils::set_start_mode( $product, $start_mode );

				$start_date = isset( $_POST['_ywpo_start_date'] ) ? sanitize_text_field( wp_unslash( $_POST['_ywpo_start_date'] ) ) : '';
				YITH_Pre_Order_Utils::set_start_date( $product, $start_date );

				$start_date_label = isset( $_POST['_ywpo_start_date_label'] ) ? wp_kses_post( wp_unslash( $_POST['_ywpo_start_date_label'] ) ) : '';
				YITH_Pre_Order_Utils::set_start_date_label( $product, $start_date_label );

				$availability_date_mode = isset( $_POST['_ywpo_availability_date_mode'] ) ? sanitize_text_field( wp_unslash( $_POST['_ywpo_availability_date_mode'] ) ) : '';
				YITH_Pre_Order_Utils::set_availability_date_mode( $product, $availability_date_mode );

				$old_release_date = YITH_Pre_Order_Utils::get_for_sale_date( $product );
				$new_release_date = (string) isset( $_POST['_ywpo_for_sale_date'] ) ? sanitize_text_field( wp_unslash( $_POST['_ywpo_for_sale_date'] ) ) : '';

				if (
					( ! empty( $old_release_date ) && ! empty( $new_release_date ) ) &&
					( $new_release_date !== $old_release_date )
				) {
					if (
						apply_filters( 'ywpo_allow_release_date_change', false, $post_id, $old_release_date, $new_release_date ) ||
						'yes' === get_option( 'ywpo_release_date_change_email_enabled', 'no' )
					) {
						$old_release_date_timestamp = YITH_Pre_Order_Utils::get_for_sale_date_timestamp( $product );
						$new_release_date_timestamp = strtotime( get_gmt_from_date( $new_release_date ) );

						WC()->mailer();
						$customer_emails = ywpo_update_release_date_itemmeta_by_product( $post_id, $old_release_date_timestamp, $new_release_date_timestamp );

						if ( $customer_emails ) {
							foreach ( $customer_emails as $email ) {
								do_action( 'ywpo_release_date_changed_email', $email, $post_id, $old_release_date_timestamp, $new_release_date_timestamp );
							}
						}
					}
				}
				YITH_Pre_Order_Utils::set_for_sale_date( $product, $new_release_date );

				$dynamic_availability_date = isset( $_POST['_ywpo_dynamic_availability_date'] ) ? sanitize_text_field( wp_unslash( $_POST['_ywpo_dynamic_availability_date'] ) ) : '';
				YITH_Pre_Order_Utils::set_dynamic_availability_date( $product, $dynamic_availability_date );

				$price_mode = isset( $_POST['_ywpo_price_mode'] ) ? sanitize_text_field( wp_unslash( $_POST['_ywpo_price_mode'] ) ) : '';
				YITH_Pre_Order_Utils::set_price_mode( $product, $price_mode );

				$pre_order_price = isset( $_POST['_ywpo_preorder_price'] ) ? wc_format_decimal( sanitize_text_field( wp_unslash( $_POST['_ywpo_preorder_price'] ) ) ) : '';
				YITH_Pre_Order_Utils::set_pre_order_price( $product, $pre_order_price );

				$discount_percentage = isset( $_POST['_ywpo_preorder_discount_percentage'] ) ? sanitize_text_field( wp_unslash( $_POST['_ywpo_preorder_discount_percentage'] ) ) : '';
				YITH_Pre_Order_Utils::set_discount_percentage( $product, $discount_percentage );
				$discount_fixed = isset( $_POST['_ywpo_preorder_discount_fixed'] ) ? sanitize_text_field( wp_unslash( $_POST['_ywpo_preorder_discount_fixed'] ) ) : '';
				YITH_Pre_Order_Utils::set_discount_fixed( $product, $discount_fixed );
				$increase_percentage = isset( $_POST['_ywpo_preorder_increase_percentage'] ) ? sanitize_text_field( wp_unslash( $_POST['_ywpo_preorder_increase_percentage'] ) ) : '';
				YITH_Pre_Order_Utils::set_increase_percentage( $product, $increase_percentage );
				$increase_fixed = isset( $_POST['_ywpo_preorder_increase_fixed'] ) ? sanitize_text_field( wp_unslash( $_POST['_ywpo_preorder_increase_fixed'] ) ) : '';
				YITH_Pre_Order_Utils::set_increase_fixed( $product, $increase_fixed );

				$max_qty_enabled = isset( $_POST['_ywpo_max_qty_enabled'] ) ? sanitize_text_field( wp_unslash( $_POST['_ywpo_max_qty_enabled'] ) ) : '';
				YITH_Pre_Order_Utils::set_max_qty_enabled( $product, $max_qty_enabled );

				$max_qty = isset( $_POST['_ywpo_max_qty'] ) ? sanitize_text_field( wp_unslash( $_POST['_ywpo_max_qty'] ) ) : '';
				YITH_Pre_Order_Utils::set_max_qty( $product, $max_qty );

				$override_labels = isset( $_POST['_ywpo_override_labels'] ) ? sanitize_text_field( wp_unslash( $_POST['_ywpo_override_labels'] ) ) : '';
				YITH_Pre_Order_Utils::set_override_labels( $product, $override_labels );

				$pre_order_label = isset( $_POST['_ywpo_preorder_label'] ) ? wp_kses_post( wp_unslash( $_POST['_ywpo_preorder_label'] ) ) : '';
				YITH_Pre_Order_Utils::set_pre_order_label( $product, $pre_order_label );

				$pre_order_availability_date_label = isset( $_POST['_ywpo_preorder_availability_date_label'] ) ? wp_kses_post( wp_unslash( $_POST['_ywpo_preorder_availability_date_label'] ) ) : '';
				YITH_Pre_Order_Utils::set_pre_order_availability_date_label( $product, $pre_order_availability_date_label );

				$pre_order_no_date_label = isset( $_POST['_ywpo_preorder_no_date_label'] ) ? wp_kses_post( wp_unslash( $_POST['_ywpo_preorder_no_date_label'] ) ) : '';
				YITH_Pre_Order_Utils::set_pre_order_no_date_label( $product, $pre_order_no_date_label );

				$override_fee = isset( $_POST['_ywpo_override_fee'] ) ? sanitize_text_field( wp_unslash( $_POST['_ywpo_override_fee'] ) ) : '';
				YITH_Pre_Order_Utils::set_override_fee( $product, $override_fee );

				$pre_order_fee = isset( $_POST['_ywpo_fee'] ) ? sanitize_text_field( wp_unslash( $_POST['_ywpo_fee'] ) ) : '';
				YITH_Pre_Order_Utils::set_fee( $product, $pre_order_fee );

				$override_charge_type = isset( $_POST['_ywpo_override_charge_type'] ) ? sanitize_text_field( wp_unslash( $_POST['_ywpo_override_charge_type'] ) ) : '';
				YITH_Pre_Order_Utils::set_override_charge_type( $product, $override_charge_type );

				$charge_type = isset( $_POST['_ywpo_charge_type'] ) ? sanitize_text_field( wp_unslash( $_POST['_ywpo_charge_type'] ) ) : '';
				YITH_Pre_Order_Utils::set_charge_type( $product, $charge_type );
			}
		}

		/**
		 * This function is executed when the variations post meta are being updated in the Edit product page.
		 *
		 * @param string|int $post_id The post's ID.
		 * @param int        $_i      Iterator.
		 */
		public function update_variation_post_meta( $post_id, $_i ) {
			// phpcs:disable WordPress.Security.NonceVerification.Missing
			$product = wc_get_product( $post_id );

			if ( ! isset( $_POST['_ywpo_preorder'][ $_i ] ) ) {
				YITH_Pre_Order_Utils::set_pre_order_status( $product, 'no' );
				return;
			}

			$is_pre_order = ! empty( $_POST['_ywpo_preorder'][ $_i ] ) ? 'yes' : 'no';
			YITH_Pre_Order_Utils::set_pre_order_status( $product, $is_pre_order );

			$pre_order_mode = isset( $_POST['_ywpo_pre_order_mode'][ $_i ] ) ? sanitize_text_field( wp_unslash( $_POST['_ywpo_pre_order_mode'][ $_i ] ) ) : '';
			if ( $pre_order_mode ) {
				YITH_Pre_Order_Utils::set_pre_order_mode( $product, $pre_order_mode );
			}

			$override_pre_order_mode = isset( $_POST['_ywpo_override_pre_order_mode'][ $_i ] ) ? sanitize_text_field( wp_unslash( $_POST['_ywpo_override_pre_order_mode'][ $_i ] ) ) : '';
			if ( $override_pre_order_mode ) {
				YITH_Pre_Order_Utils::set_override_pre_order_mode( $product, $override_pre_order_mode );
			}

			$start_mode = isset( $_POST['_ywpo_start_mode'][ $_i ] ) ? sanitize_text_field( wp_unslash( $_POST['_ywpo_start_mode'][ $_i ] ) ) : '';
			YITH_Pre_Order_Utils::set_start_mode( $product, $start_mode );

			$start_date = isset( $_POST['_ywpo_start_date'][ $_i ] ) ? sanitize_text_field( wp_unslash( $_POST['_ywpo_start_date'][ $_i ] ) ) : '';
			YITH_Pre_Order_Utils::set_start_date( $product, $start_date );

			$start_date_label = isset( $_POST['_ywpo_start_date_label'][ $_i ] ) ? wp_kses_post( wp_unslash( $_POST['_ywpo_start_date_label'][ $_i ] ) ) : '';
			YITH_Pre_Order_Utils::set_start_date_label( $product, $start_date_label );

			$availability_date_mode = isset( $_POST['_ywpo_availability_date_mode'][ $_i ] ) ? sanitize_text_field( wp_unslash( $_POST['_ywpo_availability_date_mode'][ $_i ] ) ) : '';
			YITH_Pre_Order_Utils::set_availability_date_mode( $product, $availability_date_mode );

			$old_release_date = YITH_Pre_Order_Utils::get_for_sale_date( $product );
			$new_release_date = (string) isset( $_POST['_ywpo_for_sale_date'][ $_i ] ) ? sanitize_text_field( wp_unslash( $_POST['_ywpo_for_sale_date'][ $_i ] ) ) : '';

			if (
				( ! empty( $old_release_date ) && ! empty( $new_release_date ) ) &&
				( $new_release_date !== $old_release_date )
			) {
				if (
					apply_filters( 'ywpo_allow_release_date_change', false, $post_id, $old_release_date, $new_release_date ) ||
					'yes' === get_option( 'ywpo_release_date_change_email_enabled', 'no' )
				) {
					$old_release_date_timestamp = YITH_Pre_Order_Utils::get_for_sale_date_timestamp( $product );
					$new_release_date_timestamp = strtotime( get_gmt_from_date( $new_release_date ) );

					WC()->mailer();
					$customer_emails = ywpo_update_release_date_itemmeta_by_product( $post_id, $old_release_date_timestamp, $new_release_date_timestamp );

					if ( $customer_emails ) {
						foreach ( $customer_emails as $email ) {
							do_action( 'ywpo_release_date_changed_email', $email, $post_id, $old_release_date_timestamp, $new_release_date_timestamp );
						}
					}
				}
			}
			YITH_Pre_Order_Utils::set_for_sale_date( $product, $new_release_date );

			$dynamic_availability_date = isset( $_POST['_ywpo_dynamic_availability_date'][ $_i ] ) ? sanitize_text_field( wp_unslash( $_POST['_ywpo_dynamic_availability_date'][ $_i ] ) ) : '';
			YITH_Pre_Order_Utils::set_dynamic_availability_date( $product, $dynamic_availability_date );

			$price_mode = isset( $_POST['_ywpo_price_mode'][ $_i ] ) ? sanitize_text_field( wp_unslash( $_POST['_ywpo_price_mode'][ $_i ] ) ) : '';
			YITH_Pre_Order_Utils::set_price_mode( $product, $price_mode );

			$pre_order_price = isset( $_POST['_ywpo_preorder_price'][ $_i ] ) ? wc_format_decimal( sanitize_text_field( wp_unslash( $_POST['_ywpo_preorder_price'][ $_i ] ) ) ) : '';
			YITH_Pre_Order_Utils::set_pre_order_price( $product, $pre_order_price );

			$discount_percentage = isset( $_POST['_ywpo_preorder_discount_percentage'][ $_i ] ) ? sanitize_text_field( wp_unslash( $_POST['_ywpo_preorder_discount_percentage'][ $_i ] ) ) : '';
			YITH_Pre_Order_Utils::set_discount_percentage( $product, $discount_percentage );
			$discount_fixed = isset( $_POST['_ywpo_preorder_discount_fixed'][ $_i ] ) ? sanitize_text_field( wp_unslash( $_POST['_ywpo_preorder_discount_fixed'][ $_i ] ) ) : '';
			YITH_Pre_Order_Utils::set_discount_fixed( $product, $discount_fixed );
			$increase_percentage = isset( $_POST['_ywpo_preorder_increase_percentage'][ $_i ] ) ? sanitize_text_field( wp_unslash( $_POST['_ywpo_preorder_increase_percentage'][ $_i ] ) ) : '';
			YITH_Pre_Order_Utils::set_increase_percentage( $product, $increase_percentage );
			$increase_fixed = isset( $_POST['_ywpo_preorder_increase_fixed'][ $_i ] ) ? sanitize_text_field( wp_unslash( $_POST['_ywpo_preorder_increase_fixed'][ $_i ] ) ) : '';
			YITH_Pre_Order_Utils::set_increase_fixed( $product, $increase_fixed );

			$max_qty_enabled = isset( $_POST['_ywpo_max_qty_enabled'][ $_i ] ) ? sanitize_text_field( wp_unslash( $_POST['_ywpo_max_qty_enabled'][ $_i ] ) ) : '';
			YITH_Pre_Order_Utils::set_max_qty_enabled( $product, $max_qty_enabled );

			$max_qty = isset( $_POST['_ywpo_max_qty'][ $_i ] ) ? sanitize_text_field( wp_unslash( $_POST['_ywpo_max_qty'][ $_i ] ) ) : '';
			YITH_Pre_Order_Utils::set_max_qty( $product, $max_qty );

			$override_labels = isset( $_POST['_ywpo_override_labels'][ $_i ] ) ? sanitize_text_field( wp_unslash( $_POST['_ywpo_override_labels'][ $_i ] ) ) : '';
			YITH_Pre_Order_Utils::set_override_labels( $product, $override_labels );

			$pre_order_label = isset( $_POST['_ywpo_preorder_label'][ $_i ] ) ? wp_kses_post( wp_unslash( $_POST['_ywpo_preorder_label'][ $_i ] ) ) : '';
			YITH_Pre_Order_Utils::set_pre_order_label( $product, $pre_order_label );

			$pre_order_availability_date_label = isset( $_POST['_ywpo_preorder_availability_date_label'][ $_i ] ) ? wp_kses_post( wp_unslash( $_POST['_ywpo_preorder_availability_date_label'][ $_i ] ) ) : '';
			YITH_Pre_Order_Utils::set_pre_order_availability_date_label( $product, $pre_order_availability_date_label );

			$pre_order_no_date_label = isset( $_POST['_ywpo_preorder_no_date_label'][ $_i ] ) ? wp_kses_post( wp_unslash( $_POST['_ywpo_preorder_no_date_label'][ $_i ] ) ) : '';
			YITH_Pre_Order_Utils::set_pre_order_no_date_label( $product, $pre_order_no_date_label );

			$override_fee = isset( $_POST['_ywpo_override_fee'][ $_i ] ) ? sanitize_text_field( wp_unslash( $_POST['_ywpo_override_fee'][ $_i ] ) ) : '';
			YITH_Pre_Order_Utils::set_override_fee( $product, $override_fee );

			$pre_order_fee = isset( $_POST['_ywpo_fee'][ $_i ] ) ? sanitize_text_field( wp_unslash( $_POST['_ywpo_fee'][ $_i ] ) ) : '';
			YITH_Pre_Order_Utils::set_fee( $product, $pre_order_fee );

			$override_charge_type = isset( $_POST['_ywpo_override_charge_type'][ $_i ] ) ? sanitize_text_field( wp_unslash( $_POST['_ywpo_override_charge_type'][ $_i ] ) ) : '';
			YITH_Pre_Order_Utils::set_override_charge_type( $product, $override_charge_type );

			$charge_type = isset( $_POST['_ywpo_charge_type'][ $_i ] ) ? sanitize_text_field( wp_unslash( $_POST['_ywpo_charge_type'][ $_i ] ) ) : '';
			YITH_Pre_Order_Utils::set_charge_type( $product, $charge_type );
		}

        /**
         * Convert the pre-order item meta into a human-readable format.
         *
         * @param array $formatted_meta Item meta array.
         *
         * @return array
         */
        public function format_item_meta_data( $formatted_meta ) {
            foreach ( $formatted_meta as $meta ) {
                if ( '_ywpo_item_preorder' === $meta->key ) {
                    $meta->display_key   = __( 'Pre-order item', 'yith-pre-order-for-woocommerce' );
                    $meta->display_value = __( 'Yes', 'yith-pre-order-for-woocommerce' );
                }
                if ( '_ywpo_item_status' === $meta->key ) {
                    $meta->display_key = __( 'Pre-order status', 'yith-pre-order-for-woocommerce' );
                    switch ( $meta->value ) {
                        case 'waiting':
                            $meta->display_value = __( 'Waiting', 'yith-pre-order-for-woocommerce' );
                            break;
                        case 'completed':
                            $meta->display_value = __( 'Completed', 'yith-pre-order-for-woocommerce' );
                            break;
                        case 'cancelled':
                            $meta->display_value = __( 'Cancelled', 'yith-pre-order-for-woocommerce' );
                            break;
                    }
                }
                if ( '_ywpo_item_for_sale_date' === $meta->key ) {
                    $meta->display_key   = __( 'Availability date', 'yith-pre-order-for-woocommerce' );
                    $meta->display_value = '<p>' . ywpo_print_datetime( $meta->value ) . '</p>';
                }
            }

            return $formatted_meta;
        }

        /**
         * Hide the unnecessary pre-order item meta.
         *
         * @param array $metas Item meta array.
         *
         * @return array
         */
        public function hidden_order_item_meta( $metas ) {
            $metas[] = '_ywpo_new_pre_order_email_sent';
            return $metas;
        }
	}
}

/**
 * Unique access to instance of YITH_Pre_Order_Edit_Product_Page_Premium class
 *
 * @return YITH_Pre_Order_Edit_Product_Page_Premium
 */
function YITH_Pre_Order_Edit_Product_Page_Premium() { // phpcs:ignore WordPress.NamingConventions.ValidFunctionName.FunctionNameInvalid
	return YITH_Pre_Order_Edit_Product_Page_Premium::get_instance();
}
