<?php
/**
 * This file belongs to the YIT Framework.
 *
 * This source file is subject to the GNU GENERAL PUBLIC LICENSE (GPL 3.0)
 * that is bundled with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * http://www.gnu.org/licenses/gpl-3.0.txt
 *
 * @package YITH\PreOrder\Includes\Emails
 * @author YITH <plugins@yithemes.com>
 */

if ( ! defined( 'YITH_WCPO_VERSION' ) ) {
	exit( 'Direct access forbidden.' );
}

if ( ! class_exists( 'YITH_Pre_Order_Out_Of_Stock_Email' ) ) {
	/**
	 * Class YITH_Pre_Order_Out_Of_Stock_Email
	 */
	class YITH_Pre_Order_Out_Of_Stock_Email extends YITH_Pre_Order_Email {

		/**
		 * Constructor.
		 */
		public function __construct() {

			$this->id = 'ywpo_out_of_stock_email';

			$this->title       = __( 'YITH Pre-Order: An out-of-stock product became pre-order', 'yith-pre-order-for-woocommerce' );
			$this->description = __( 'The administrator will receive an email when a product goes out of stock and becomes a pre-order product.', 'yith-pre-order-for-woocommerce' );
			$this->heading     = __( 'A product from your shop is now in pre-order mode', 'yith-pre-order-for-woocommerce' );
			$this->subject     = __( 'A product from your shop is now in pre-order mode', 'yith-pre-order-for-woocommerce' );

			$this->email_body = __(
				"Hi admin,\n\nWe would like to inform you that <strong>a product from your shop is out of stock</strong> and is now in pre-order mode:
{product_table}\nThe pre-order mode will be automatically disabled when the product is back in stock or if you choose to manually manage the pre-order options.
\nRegards,\n{site_title}",
				'yith-pre-order-for-woocommerce'
			);

			$this->template_html = 'emails/ywpo-email-admin-out-of-stock.php';

			add_action( 'ywpo_out_of_stock_email', array( $this, 'trigger' ) );

			parent::__construct();

			$this->recipient  = $this->get_option( 'recipient', get_option( 'admin_email' ) );
			$this->email_type = 'html';
		}

		/**
		 * Trigger the sending of this email.
		 *
		 * @param WC_Product $product The WC_Product object.
		 */
		public function trigger( $product ) {
			if ( ! $product instanceof WC_Product ) {
				return;
			}

			$this->data = apply_filters(
				'ywpo_out_of_stock_email_object',
				array(
					'product' => $product,
				),
				$product,
			);

			$product_url = get_admin_url( null, 'post.php?post=' . $product->get_id() . '&action=edit' );

			$product_link = apply_filters(
				'ywpo_out_of_stock_email_product_link',
				'<a href="' . $product_url . '">' . $product->get_title() . '</a>',
				$product
			);

			$this->placeholders['{product_title}'] = $product->get_title();
			$this->placeholders['{product_url}']   = $product_url;
			$this->placeholders['{product_link}']  = $product_link;

			$this->email_body = $this->format_string( $this->get_option( 'email_body', $this->email_body ) );

			if ( $this->is_enabled() && $this->get_recipient() ) {
				$this->send( $this->get_recipient(), $this->get_subject(), $this->get_content(), $this->get_headers(), $this->get_attachments() );
			}

		}

		/**
		 * Get content html.
		 *
		 * @return string
		 */
		public function get_content_html() {
			if ( has_filter( 'woocommerce_is_email_preview' ) ) {
				$this->data       = array( 'product' => $this->get_dummy_product( true ) );
				$this->email_body = $this->format_string( $this->get_option( 'email_body', $this->email_body ) );
			}
			return wc_get_template_html(
				$this->template_html,
				array(
					'email_heading' => $this->get_heading(),
					'sent_to_admin' => true,
					'plain_text'    => false,
					'email'         => $this,
				),
				'',
				YITH_WCPO_TEMPLATE_PATH
			);
		}

		/**
		 * Initialise Settings Form Fields - these are generic email options most will use.
		 */
		public function init_form_fields() {
			$this->form_fields = array(
				'enabled'    => array(
					'title'   => esc_html__( 'Enable/Disable', 'yith-pre-order-for-woocommerce' ),
					'type'    => 'checkbox',
					'label'   => esc_html__( 'Enable this email notification', 'yith-pre-order-for-woocommerce' ),
					'default' => 'yes',
				),
				'recipient'  => array(
					'title'       => __( 'Recipient(s)', 'yith-pre-order-for-woocommerce' ),
					'type'        => 'text',
					// translators: %s: email recipient.
					'description' => sprintf( __( 'Enter recipients (comma separated) for this email. Defaults to %s.', 'yith-pre-order-for-woocommerce' ), '<code>' . esc_attr( get_option( 'admin_email' ) ) . '</code>' ),
					'placeholder' => '',
					'default'     => '',
				),
				'subject'    => array(
					'title'       => __( 'Subject', 'yith-pre-order-for-woocommerce' ),
					'type'        => 'text',
					// translators: %s: email subject.
					'description' => sprintf( __( 'This controls the email subject line. Leave blank to use the default subject: <code>%s</code>.', 'yith-pre-order-for-woocommerce' ), $this->subject ),
					'placeholder' => '',
					'default'     => $this->subject,
				),
				'heading'    => array(
					'title'       => __( 'Email heading', 'yith-pre-order-for-woocommerce' ),
					'type'        => 'text',
					// translators: %s: email heading.
					'description' => sprintf( __( 'This controls the main heading included in the email notification. Leave blank to use the default heading: <code>%s</code>.', 'yith-pre-order-for-woocommerce' ), $this->heading ),
					'placeholder' => '',
					'default'     => $this->heading,
				),
				'email_body' => array(
					'title'       => __( 'Email body', 'yith-pre-order-for-woocommerce' ),
					'type'        => 'textarea',
					// translators: %s: email body.
					'description' => sprintf( __( 'Defaults to <code>%s</code>', 'yith-pre-order-for-woocommerce' ), $this->email_body ),
					'placeholder' => '',
					'default'     => $this->email_body,
				),
			);
		}
	}
}

return new YITH_Pre_Order_Out_Of_Stock_Email();
