<?php
/**
 * This file belongs to the YIT Framework.
 *
 * This source file is subject to the GNU GENERAL PUBLIC LICENSE (GPL 3.0)
 * that is bundled with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * http://www.gnu.org/licenses/gpl-3.0.txt
 *
 * @package YITH\PreOrder\Includes\Emails
 * @author YITH <plugins@yithemes.com>
 */

if ( ! defined( 'YITH_WCPO_VERSION' ) ) {
	exit( 'Direct access forbidden.' );
}

if ( ! class_exists( 'YITH_Pre_Order_Release_Date_Reminder_Email' ) ) {
	/**
	 * Class YITH_Pre_Order_Release_Date_Reminder_Email
	 */
	class YITH_Pre_Order_Release_Date_Reminder_Email extends YITH_Pre_Order_Email {

		/**
		 * Constructor.
		 */
		public function __construct() {
			$this->id = 'ywpo_release_date_reminder';

			$this->title       = __( 'YITH Pre-Order: Availability date is near', 'yith-pre-order-for-woocommerce' );
			$this->description = sprintf(
			// translators: %s: placeholders.
				__( 'The administrator will receive an email before the availability date is reached. Available placeholders: %s', 'yith-pre-order-for-woocommerce' ),
				'<code>{product_title}, {product_url}, {product_link}, {release_date}, {release_datetime}, {offset}, {site_title}.</code>'
			);

			$this->heading    = __( "This product's availability date is near!", 'yith-pre-order-for-woocommerce' );
			$this->subject    = __( "This product's availability date is near!", 'yith-pre-order-for-woocommerce' );
			$this->email_body = __(
				"Hi admin,\n\nWe would like to inform you this product's availability date is near and the pre-order mode is going to be disabled:
{product_table}\nWhen the pre-order mode ends, this product will be available for sale in your shop and the customers that pre-ordered it will receive an email confirming the product is now available.
\n<strong>Not ready for that?</strong>\nDon't worry. You can <a>change the availability date by editing the product.</a>
\nRegards,\n{site_title}",
				'yith-pre-order-for-woocommerce'
			);

			$this->template_html = 'emails/ywpo-email-admin-release-date-reminder.php';

			add_action( 'ywpo_release_date_reminder', array( $this, 'trigger' ) );

			parent::__construct();

			$this->email_type = 'html';
			$this->recipient  = $this->get_option( 'recipient', get_option( 'admin_email' ) );

		}

		/**
		 * Triggers the email sending process.
		 *
		 * @param WC_Product $product The WC_Product object.
		 */
		public function trigger( $product ) {
			if ( ! $product instanceof WC_Product ) {
				return;
			}

			$this->data = apply_filters(
				'ywpo_release_date_reminder_email_object',
				array(
					'product' => $product,
				),
				$product,
			);

			$product_url = get_admin_url( null, 'post.php?post=' . $product->get_id() . '&action=edit' );

			$product_link = apply_filters(
				'ywpo_release_date_reminder_email_product_link',
				'<a href="' . $product_url . '">' . $product->get_title() . '</a>',
				$product
			);

			$release_date        = ywpo_get_release_date( $product );
			$no_release_date_msg = apply_filters( 'ywpo_release_date_reminder_email_no_release_date', __( 'No date', 'yith-pre-order-for-woocommerce' ), $product );

			$this->placeholders['{product_title}']    = $product->get_title();
			$this->placeholders['{product_url}']      = $product_url;
			$this->placeholders['{product_link}']     = $product_link;
			$this->placeholders['{release_date}']     = $release_date ? ywpo_print_date( $release_date ) : $no_release_date_msg;
			$this->placeholders['{release_datetime}'] = $release_date ? ywpo_print_datetime( $release_date ) : $no_release_date_msg;
			$this->placeholders['{offset}']           = ywpo_get_timezone_offset_label();

			$this->email_body = $this->format_string( $this->get_option( 'email_body', $this->email_body ) );

			if ( $this->is_enabled() && $this->get_recipient() ) {
				$return = $this->send(
					$this->get_recipient(),
					$this->get_subject(),
					$this->get_content(),
					$this->get_headers(),
					$this->get_attachments()
				);
				if ( apply_filters( 'ywpo_release_date_reminder_email_apply_notified_meta', $return, $product, $this ) ) {
					update_post_meta( $product->get_id(), '_ywpo_preorder_notified', 'yes' );
				}
			}
		}

		/**
		 * Get content html.
		 *
		 * @return string
		 */
		public function get_content_html() {
			if ( has_filter( 'woocommerce_is_email_preview' ) ) {
				$this->data = array( 'product' => $this->get_dummy_product() );
			}
			return wc_get_template_html(
				$this->template_html,
				array(
					'email_heading' => $this->get_heading(),
					'sent_to_admin' => true,
					'plain_text'    => false,
					'email'         => $this,
				),
				'',
				YITH_WCPO_TEMPLATE_PATH
			);
		}

		/**
		 * Initialise Settings Form Fields - these are generic email options most will use.
		 */
		public function init_form_fields() {
			$this->form_fields = array(
				'enabled'    => array(
					'title'   => __( 'Enable/Disable', 'yith-pre-order-for-woocommerce' ),
					'type'    => 'checkbox',
					'label'   => __( 'Enable this email notification', 'yith-pre-order-for-woocommerce' ),
					'default' => 'yes',
				),
				'recipient'  => array(
					'title'       => __( 'Recipient(s)', 'yith-pre-order-for-woocommerce' ),
					'type'        => 'text',
					// translators: %s: email recipient.
					'description' => sprintf( __( 'Enter recipients (comma separated) for this email. Defaults to %s.', 'yith-pre-order-for-woocommerce' ), '<code>' . esc_attr( get_option( 'admin_email' ) ) . '</code>' ),
					'placeholder' => '',
					'default'     => '',
				),
				'subject'    => array(
					'title'       => __( 'Subject', 'yith-pre-order-for-woocommerce' ),
					'type'        => 'text',
					// translators: %s: email subject.
					'description' => sprintf( __( 'This controls the email subject line. Leave blank to use the default subject: <code>%s</code>.', 'yith-pre-order-for-woocommerce' ), $this->subject ),
					'placeholder' => '',
					'default'     => $this->subject,
				),
				'heading'    => array(
					'title'       => __( 'Email heading', 'yith-pre-order-for-woocommerce' ),
					'type'        => 'text',
					// translators: %s: email heading.
					'description' => sprintf( __( 'This controls the main heading included in the email notification. Leave blank to use the default heading: <code>%s</code>.', 'yith-pre-order-for-woocommerce' ), $this->heading ),
					'placeholder' => '',
					'default'     => $this->heading,
				),
				'email_body' => array(
					'title'       => __( 'Email body', 'yith-pre-order-for-woocommerce' ),
					'type'        => 'textarea',
					// translators: %s: email body.
					'description' => sprintf( __( 'Defaults to <code>%s</code>', 'yith-pre-order-for-woocommerce' ), $this->email_body ),
					'placeholder' => '',
					'default'     => $this->email_body,
				),
			);
		}
	}
}

return new YITH_Pre_Order_Release_Date_Reminder_Email();
