<?php
/**
 * This file belongs to the YIT Framework.
 *
 * This source file is subject to the GNU GENERAL PUBLIC LICENSE (GPL 3.0)
 * that is bundled with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * http://www.gnu.org/licenses/gpl-3.0.txt
 *
 * @package YITH\PreOrder\Includes
 * @author YITH <plugins@yithemes.com>
 */

use Automattic\WooCommerce\Internal\Admin\Orders\ListTable;
use Automattic\WooCommerce\Utilities\OrderUtil;

if ( ! defined( 'YITH_WCPO_VERSION' ) ) {
	exit( 'Direct access forbidden.' );
}

if ( ! class_exists( 'YITH_Pre_Order_Admin_Premium' ) ) {
	/**
	 * Class YITH_Pre_Order_Admin_Premium
	 */
	class YITH_Pre_Order_Admin_Premium extends YITH_Pre_Order_Admin {

		/**
		 * One week in timestamp format.
		 *
		 * @var int $one_week
		 */
		private static $one_week = 604800;

		/**
		 * The post type used for the current List Table
		 *
		 * @var string $list_table_post_type
		 */
		private $list_table_post_type = '';

		/**
		 * Returns single instance of the class
		 *
		 * @return YITH_Pre_Order_Admin_Premium
		 * @since 1.0.0
		 */
		public static function get_instance() {
			if ( is_null( self::$instance ) ) {
				self::$instance = new self();
			}

			return self::$instance;
		}

		/**
		 * Construct
		 *
		 * @since 1.0
		 */
		public function __construct() {
			$this->show_premium_landing = false;
			$this->edit_product_page    = YITH_Pre_Order_Edit_Product_Page_Premium();

			parent::__construct();

			add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_scripts' ), 20 );
			add_filter( 'woocommerce_email_actions', array( $this, 'add_for_sale_date_changed_trigger_action' ) );
			if ( OrderUtil::custom_orders_table_usage_is_enabled() ) {
				add_filter( 'woocommerce_shop_order_list_table_columns', array( $this, 'add_pre_order_column' ) );
				add_action( 'woocommerce_shop_order_list_table_custom_column', array( $this, 'add_pre_order_column_content_orders' ), 10, 2 );
			} else {
				add_filter( 'manage_edit-product_columns', array( $this, 'add_pre_order_column' ) );
				add_action( 'manage_product_posts_custom_column', array( $this, 'add_pre_order_column_content_products' ), 10, 2 );
				add_filter( 'manage_edit-shop_order_columns', array( $this, 'add_pre_order_column' ) );
				add_action( 'manage_shop_order_posts_custom_column', array( $this, 'add_pre_order_column_content_orders' ), 10, 2 );
			}
			add_action( 'ywpo_list_table_products', array( $this, 'list_table_products' ) );
			add_action( 'ywpo_list_table_orders', array( $this, 'list_table_orders' ) );
			add_action( 'admin_action_ywpo_export_pre_order_products', array( $this, 'export_pre_order_products_via_csv' ) );
			add_action( 'admin_action_ywpo_export_pre_orders', array( $this, 'export_pre_orders_via_csv' ) );
			add_action( 'yith_plugin_fw_get_field_after', array( $this, 'print_field_after_html' ), 10, 1 );
			add_action( 'wp_ajax_ywpo_manage_products_modal_content', array( $this, 'manage_products_modal_content' ) );
			add_filter( 'handle_bulk_actions-edit-product', array( $this, 'handle_product_bulk_actions' ), 10, 3 );
			add_action( 'load-edit-tags.php', array( $this, 'handle_taxonomy_bulk_actions' ) );
			add_action( 'load-yith-plugins_page_yith_wcpo_panel', array( $this, 'add_posts_per_page_screen_option' ) );
		}

		/**
		 * Enqueue the scripts for admin pages.
		 */
		public function enqueue_scripts() {
			$current_screen = get_current_screen();
			if ( 'yith-plugins_page_yith_wcpo_panel' === $current_screen->id ) {
				$integrated_gateways = ywpo_exist_integrated_gateways();
				wp_register_script( 'ywpo-options', YITH_WCPO_ASSETS_JS_URL . yit_load_js_file( 'ywpo-options.js' ), array( 'jquery' ), YITH_WCPO_VERSION, 'true' );
				wp_localize_script( 'ywpo-options', 'ywpo_options', array( 'integrated_gateways' => $integrated_gateways ) );
				wp_enqueue_script( 'ywpo-options' );
				wp_enqueue_style( 'ywpo-admin', YITH_WCPO_ASSETS_URL . 'css/ywpo-admin.css', array(), YITH_WCPO_VERSION );

				$vars = $_GET; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
				if (
					( isset( $vars['page'] ) && 'yith_wcpo_panel' === $vars['page'] && ! isset( $vars['tab'] ) ) ||
					( isset( $vars['tab'] ) && 'pre-orders' === $vars['tab'] )
				) {
					wp_register_script(
						'ywpo-pre-order-list-table',
						YITH_WCPO_ASSETS_JS_URL . yit_load_js_file( 'ywpo-pre-order-list-table.js' ),
						array( 'jquery', 'jquery-blockui', 'jquery-tiptip' ),
						YITH_WCPO_VERSION,
						false
					);
					$params = array(
						'ajax_url'             => admin_url( 'admin-ajax.php', apply_filters( 'ywpo_ajax_url_scheme_backend', '' ) ),
						'create_modal_nonce'   => wp_create_nonce( 'manage-products-modal-content' ),
						'item_action_nonce'    => wp_create_nonce( 'manage-products-modal-item-action' ),
						'title'                => apply_filters( 'ywpo_manage_products_modal_title', __( 'Manage products in pre-order #', 'yith-pre-order-for-woocommerce' ) ),
						'confirm_text'         => apply_filters( 'ywpo_manage_products_modal_confirm_button_text', __( 'Save changes', 'yith-pre-order-for-woocommerce' ) ),
						'close_text'           => apply_filters( 'ywpo_manage_products_modal_close_text', _x( 'Close', 'Text button for closing modal window', 'yith-pre-order-for-woocommerce' ) ),
						'width'                => apply_filters( 'ywpo_manage_products_modal_width', 800 ),
						'completed_text'       => apply_filters( 'ywpo_manage_products_modal_completed_text', __( 'Completed', 'yith-pre-order-for-woocommerce' ) ),
						'cancelled_text'       => apply_filters( 'ywpo_manage_products_modal_cancelled_text', __( 'Cancelled', 'yith-pre-order-for-woocommerce' ) ),
						'successful_completed' => apply_filters( 'ywpo_manage_products_modal_successful_completed_text', __( 'Pre-order completed successfully', 'yith-pre-order-for-woocommerce' ) ),
						'successful_cancelled' => apply_filters( 'ywpo_manage_products_modal_cancelled_text', __( 'Pre-order cancelled successfully', 'yith-pre-order-for-woocommerce' ) ),
						'complete_footer'      => wc_get_template_html( '/admin/ywpo-manage-products-modal-footer.php', array( 'type' => 'complete' ), '', YITH_WCPO_TEMPLATE_PATH ),
						'cancel_footer'        => wc_get_template_html( '/admin/ywpo-manage-products-modal-footer.php', array( 'type' => 'cancel' ), '', YITH_WCPO_TEMPLATE_PATH ),
					);
					wp_localize_script( 'ywpo-pre-order-list-table', 'ywpo_manage_products_modal', $params );
					wp_enqueue_script( 'ywpo-pre-order-list-table' );
				}
			}

			if ( 'edit-product' === $current_screen->id || 'edit-product_cat' === $current_screen->id || 'edit-product_tag' === $current_screen->id ) {
				wp_register_script(
					'ywpo-bulk-actions',
					YITH_WCPO_ASSETS_JS_URL . yit_load_js_file( 'ywpo-bulk-actions.js' ),
					array(
						'jquery',
					),
					YITH_WCPO_VERSION,
					true
				);
				wp_localize_script(
					'ywpo-bulk-actions',
					'ywpo_bk_data',
					array(
						'set_pre_order_status'    => '<option value="ywpo-set-pre-order">' . esc_html__( 'Set the pre-order status', 'yith-pre-order-for-woocommerce' ) . '</option>',
						'remove_pre_order_status' => '<option value="ywpo-remove-pre-order">' . esc_html__( 'Remove the pre-order status', 'yith-pre-order-for-woocommerce' ) . '</option>',
					)
				);
				wp_enqueue_script( 'ywpo-bulk-actions' );
			}
		}

		/**
		 * Add a panel under YITH Plugins tab
		 *
		 * @return   void
		 * @since    1.0
		 * @use      /Yit_Plugin_Panel class
		 * @see      plugin-fw/lib/yit-plugin-panel.php
		 */
		public function register_panel() {

			if ( ! empty( $this->panel ) ) {
				return;
			}

			$menu_title = 'Pre-Order';

			$admin_tabs = apply_filters(
				'yith_wcpo_admin_tabs',
				array(
					'pre-orders'    => __( 'Pre-Orders', 'yith-pre-order-for-woocommerce' ),
					'general'       => __( 'General Options', 'yith-pre-order-for-woocommerce' ),
					'payment'       => __( 'Payment Options', 'yith-pre-order-for-woocommerce' ),
					'notifications' => __( 'Notifications', 'yith-pre-order-for-woocommerce' ),
					'style'         => __( 'Style', 'yith-pre-order-for-woocommerce' ),
				)
			);

			$args = array(
				'create_menu_page' => true,
				'parent_slug'      => '',
				'plugin_slug'      => YITH_WCPO_SLUG,
				'page_title'       => 'YITH Pre-Order for WooCommerce',
				'menu_title'       => $menu_title,
				'capability'       => 'manage_options',
				'parent'           => '',
				'parent_page'      => 'yith_plugin_panel',
				'page'             => $this->panel_page,
				'admin-tabs'       => $admin_tabs,
				'class'            => yith_set_wrapper_class(),
				'options-path'     => YITH_WCPO_OPTIONS_PATH,
				'help_tab'         => array(
					'hc_url' => 'https://support.yithemes.com/hc/en-us/categories/360001024013-Plugins-Knowledge',
					'main_video' => array(
						'desc' => _x( 'Check this video to learn how to <b>configure the plugin:</b>', '[HELP TAB] Video title', 'yith-pre-order-for-woocommerce' ),
						'url'  => array(
							'en' => 'https://www.youtube.com/embed/5DbnYEHsc_0',
							'it' => 'https://www.youtube.com/embed/iZ0CaavsjSo',
							'es' => 'https://www.youtube.com/embed/Ve_nYJ7eUXU',
						),
					),
					'playlists'  => array(
						'en' => 'https://www.youtube.com/watch?v=5DbnYEHsc_0&list=PLDriKG-6905kiDTyuxpTQ8jVRKWY1FDFf',
						'it' => 'https://www.youtube.com/watch?v=iZ0CaavsjSo&list=PL9c19edGMs09w2FUr9a3f6mYT8uU9kqbk&ab_channel=YITHITALIA',
						'es' => 'https://www.youtube.com/watch?v=Ve_nYJ7eUXU&list=PL9Ka3j92PYJMuwqCiI-ijJz42mAZWgo60&ab_channel=YITHESPA%C3%91A',
					),
				),
			);

			/* === Fixed: not updated theme/old plugin framework  === */
			if ( ! class_exists( 'YIT_Plugin_Panel_WooCommerce' ) ) {
				require_once YITH_WCPO_PATH . '/plugin-fw/lib/yit-plugin-panel-wc.php';
			}

			$this->panel = new YIT_Plugin_Panel_WooCommerce( $args );
		}

		/**
		 * Plugin_row_meta
		 *
		 * Add the action links to plugin admin page.
		 *
		 * @param array  $new_row_meta_args .
		 * @param array  $plugin_meta .
		 * @param string $plugin_file .
		 * @param array  $plugin_data .
		 * @param string $status .
		 * @param string $init_file .
		 *
		 * @return   array
		 * @use      plugin_row_meta
		 */
		public function plugin_row_meta( $new_row_meta_args, $plugin_meta, $plugin_file, $plugin_data, $status, $init_file = 'YITH_WCPO_INIT' ) {
			$new_row_meta_args = parent::plugin_row_meta( $new_row_meta_args, $plugin_meta, $plugin_file, $plugin_data, $status, $init_file );

			if ( defined( $init_file ) && constant( $init_file ) === $plugin_file ) {
				$new_row_meta_args['is_premium'] = true;
			}

			return $new_row_meta_args;
		}

		/**
		 * Locate core template file
		 *
		 * @param string $core_file Template full path.
		 * @param string $template Template in use.
		 *
		 * @return string
		 * @since  1.0.0
		 */
		public function locate_core_template( $core_file, $template ) {
			$custom_template = array(
				'emails/ywpo-email-admin-new-pre-order',
				'emails/ywpo-email-admin-out-of-stock.php',
				'emails/ywpo-email-admin-release-date-reminder.php',
				'emails/ywpo-email-user-payment-reminder.php',
				'emails/ywpo-email-user-pre-order-cancelled.php',
				'emails/ywpo-email-user-pre-order-completed.php',
				'emails/ywpo-email-user-pre-order-confirmed.php',
				'emails/ywpo-email-user-release-date-changed.php',
			);

			if ( in_array( $template, $custom_template, true ) ) {
				$core_file = YITH_WCPO_TEMPLATE_PATH . $template;
			}

			return $core_file;
		}

		/**
		 * Add email action.
		 *
		 * @param array $actions Actions array.
		 *
		 * @return array
		 */
		public function add_for_sale_date_changed_trigger_action( $actions ) {
			$actions[] = 'yith_for_sale_date';
			return $actions;
		}

		/**
		 * Add the pre-order column.
		 *
		 * @param array $columns_array Columns array.
		 *
		 * @return array
		 */
		public function add_pre_order_column( $columns_array ) {
			if ( OrderUtil::custom_orders_table_usage_is_enabled() ) {
				//phpcs:ignore WordPress.Security.NonceVerification.Recommended
				if ( isset( $_GET['page'] ) && 'wc-orders' === $_GET['page'] ) {
					$columns_array['pre_order'] = esc_html__( 'Pre-Order', 'yith-pre-order-for-woocommerce' );
				}
			} else {
				//phpcs:ignore WordPress.Security.NonceVerification.Recommended
				if ( isset( $_GET['post_type'] ) && 'shop_order' === $_GET['post_type'] ) {
					$columns_array['pre_order'] = esc_html__( 'Pre-Order', 'yith-pre-order-for-woocommerce' );
				}
			}

			//phpcs:ignore WordPress.Security.NonceVerification.Recommended
			if ( isset( $_GET['post_type'] ) && 'product' === $_GET['post_type'] ) {
				$columns_array['ywpo_product'] = esc_html__( 'Pre-Order', 'yith-pre-order-for-woocommerce' );
			}
			return $columns_array;
		}

		/**
		 * Add the pre-order column in Orders page.
		 *
		 * @param string     $column_name Column name.
		 * @param string|int $post_id The post ID.
		 */
		public function add_pre_order_column_content_orders( $column_name, $post_id ) {
			if ( 'pre_order' === $column_name ) {
				$order              = wc_get_order( $post_id );
				$order_has_preorder = ywpo_order_has_pre_order( $order );
				if ( ! $order_has_preorder || 'cancelled' === $order->get_status() ) {
					return;
				}
				$pre_order_dates = array();
				foreach ( $order->get_items() as $item ) {
					if ( ! ( $item instanceof WC_Order_Item_Product ) ) {
						continue;
					}
					$product = $item->get_product();
					if ( ! ( $product instanceof WC_Product ) ) {
						continue;
					}
					$is_pre_order = ! empty( $item['ywpo_item_preorder'] ) ? $item['ywpo_item_preorder'] : '';
					$timestamp    = ! empty( $item['ywpo_item_for_sale_date'] ) ? $item['ywpo_item_for_sale_date'] : '';
					if ( 'yes' === $is_pre_order && ( $timestamp > time() || empty( $timestamp ) ) ) {
						$array_data['date']       = $timestamp;
						$array_data['product_id'] = yit_get_product_id( $product );
						$pre_order_dates[]        = $array_data;
					}
				}
				if ( $pre_order_dates ) {
					sort( $pre_order_dates );
					$showing_dates = array();
					foreach ( $pre_order_dates as $date ) {
						$array_data               = array();
						$array_data['product_id'] = $date['product_id'];
						if ( empty( $date['date'] ) ) {
							$array_data['date'] = esc_html__( 'No date set', 'yith-pre-order-for-woocommerce' );
						} else {
							$array_data['date'] = get_date_from_gmt( gmdate( 'Y-m-d H:i:s', (int) $date['date'] ), apply_filters( 'yith_ywpo_date_format', 'Y/m/d H:i', $post_id ) );
						}

						if ( empty( $date['date'] ) ) {
							$array_data['icon_color'] = 'pre_order_no_date';
						} elseif ( time() > $date['date'] ) {
							$array_data['icon_color'] = 'pre_order_finished';
						} elseif ( time() > $date['date'] - self::$one_week ) {
							$array_data['icon_color'] = 'pre_order_last_week';
						} else {
							$array_data['icon_color'] = '';
						}
						$showing_dates[] = $array_data;
					}
					foreach ( $showing_dates as $date ) {
						$product = wc_get_product( $date['product_id'] );
						if ( ! $product ) {
							continue;
						}
						if ( 'pre_order_no_date' === $date['icon_color'] ) {
							$msg = __( 'This pre-order product has no availability date.', 'yith-pre-order-for-woocommerce' );
						} elseif ( 'pre_order_finished' === $date['icon_color'] ) {
							$msg = __( 'This pre-order availability date has passed.', 'yith-pre-order-for-woocommerce' );
						} elseif ( 'pre_order_last_week' === $date['icon_color'] ) {
							$msg = __( 'This pre-order expires in a week or less.', 'yith-pre-order-for-woocommerce' );
						} else {
							$msg = __( 'Pre-order availability date.', 'yith-pre-order-for-woocommerce' );
						}
						?>
						<div class="pre_order_icon_orders_table <?php echo esc_attr( $date['icon_color'] ); ?>"
							title="<?php echo esc_attr( $msg ); ?>">
							<a href="<?php echo esc_url( get_edit_post_link( yit_get_product_id( $product ) ) ); ?>"
							title="<?php echo esc_attr( $product->get_title() ); ?>"><?php echo wp_kses_post( $product->get_formatted_name() ); ?></a>
							<?php echo ' - ' . esc_html( $date['date'] ); ?>
						</div>
						<?php
					}
				} else {
					echo '<div>' . esc_html__( 'The availability dates of all the pre-order products have passed.', 'yith-pre-order-for-woocommerce' ) . '</div>';
				}
			}
		}

		/**
		 * Add pre-order column content.
		 *
		 * @param string     $column_name Column name.
		 * @param string|int $post_id The post ID.
		 */
		public function add_pre_order_column_content_products( $column_name, $post_id ) {
			if ( 'ywpo_product' === $column_name && $post_id ) {
				$product = wc_get_product( $post_id );
				if ( ! $product instanceof WC_Product ) {
					return;
				}
				if ( 'variable' === $product->get_type() && $product->has_child() ) {
					$variations = $product->get_children();
					foreach ( $variations as $variation_id ) {
						$pre_order_variation = ywpo_get_pre_order( $variation_id );
						if ( 'yes' === $pre_order_variation->get_pre_order_status() ) {
							?>
							<div>
								<img alt="" src="<?php echo esc_attr( YITH_WCPO_ASSETS_URL . 'images/check-circle.png' ); ?>">
								<span class="ywpo_pre_order_check_icon"><?php esc_html_e( 'One or more variations', 'yith-pre-order-for-woocommerce' ); ?></span>
							</div>
							<?php
							break;
						}
					}
				} elseif ( 'simple' === $product->get_type() && YITH_Pre_Order()::is_pre_order_product( $product ) ) {
					$pre_order = ywpo_get_pre_order( $post_id );
					echo '<img src="' . esc_url( YITH_WCPO_ASSETS_URL ) . 'images/check-circle.png">';
					if ( metadata_exists( 'post', $post_id, '_ywpo_availability_date_mode' ) ) {
						if ( 'date' === $pre_order->get_availability_date_mode() ) {
							echo '<span class="ywpo_pre_order_check_icon">' . esc_html( $pre_order->get_for_sale_date() ) . '</span>';
						} elseif ( 'no_date' === $pre_order->get_availability_date_mode() ) {
							echo '<span class="ywpo_pre_order_check_icon">' . esc_html__( 'No date', 'yith-pre-order-for-woocommerce' ) . '</span>';
						}
					} elseif ( $pre_order->get_for_sale_date() ) {
						echo '<span class="ywpo_pre_order_check_icon">' . esc_html( $pre_order->get_for_sale_date() ) . '</span>';
					} else {
						echo '<span class="ywpo_pre_order_check_icon">' . esc_html__( 'No date', 'yith-pre-order-for-woocommerce' ) . '</span>';
					}
				}
			}
		}

		/**
		 * Print the template content for the Manage Products modal window.
		 */
		public function manage_products_modal_content() {
			check_ajax_referer( 'manage-products-modal-content', 'security' );
			$order_id = ! empty( $_POST['order_id'] ) ? sanitize_text_field( wp_unslash( $_POST['order_id'] ) ) : false;

			$modal_content = wc_get_template_html( '/admin/ywpo-manage-products-modal.php', array( 'order_id' => $order_id ), '', YITH_WCPO_TEMPLATE_PATH );
			wp_send_json_success( array( 'message' => $modal_content ) );
		}

		/**
		 * Display extra text next to the number field in the options that uses the parameter 'yith-wcpo-after-html'.
		 * See plugin-options/notifications-options.php
		 *
		 * @param array $field Option.
		 */
		public function print_field_after_html( $field ) {
			if ( ! empty( $field['yith-wcpo-after-html'] ) ) {
				echo '<span class="yith-wcpo-plugin-fw-field__after-html">' . esc_html( $field['yith-wcpo-after-html'] ) . '</span>';
			}
		}

		/**
		 * Export pre-order products handler.
		 */
		public function export_pre_order_products_via_csv() {
			$args = array(
				'limit'          => -1,
				'_ywpo_preorder' => 'yes',
			);

			$pre_orders = wc_get_products( $args );
			$columns    = apply_filters(
				'ywpo_pre_order_products_export_columns',
				array(
					'id'            => __( 'Product ID', 'yith-pre-order-for-woocommerce' ),
					'name'          => __( 'Name', 'yith-pre-order-for-woocommerce' ),
					'price'         => __( 'Price', 'yith-pre-order-for-woocommerce' ),
					'regular_price' => __( 'Regular price', 'yith-pre-order-for-woocommerce' ),
					'stock'         => __( 'Stock', 'yith-pre-order-for-woocommerce' ),
					'stock_status'  => __( 'Stock status', 'yith-pre-order-for-woocommerce' ),
					'downloadable'  => __( 'Downloadable', 'yith-pre-order-for-woocommerce' ),
					'virtual'       => __( 'Virtual', 'yith-pre-order-for-woocommerce' ),
				)
			);
			$this->export_csv( $pre_orders, $columns );
		}

		/**
		 * Export pre-order orders handler.
		 */
		public function export_pre_orders_via_csv() {
			if ( OrderUtil::custom_orders_table_usage_is_enabled() ) {
				$args = array(
					'limit'              => -1,
					'meta_query'     => array( // phpcs:ignore
						array(
							'key'   => '_order_has_preorder',
							'value' => 'yes',
						),
					),
				);
			} else {
				$args = array(
					'limit'              => - 1,
					'order_has_preorder' => 'yes',
				);
			}

			$pre_orders = wc_get_orders( $args );
			$columns    = apply_filters(
				'ywpo_pre_orders_export_columns',
				array(
					'id'                  => __( 'Order ID', 'yith-pre-order-for-woocommerce' ),
					'order_number'        => __( 'Order number', 'yith-pre-order-for-woocommerce' ),
					'order_date'          => __( 'Order date', 'yith-pre-order-for-woocommerce' ),
					'_order_has_preorder' => __( 'Pre-Order', 'yith-pre-order-for-woocommerce' ),
					'status'              => __( 'Status', 'yith-pre-order-for-woocommerce' ),
					'user_id'             => __( 'Customer ID', 'yith-pre-order-for-woocommerce' ),
					'currency'            => __( 'Currency', 'yith-pre-order-for-woocommerce' ),
					'prices_include_tax'  => __( 'Prices include tax', 'yith-pre-order-for-woocommerce' ),
					'total'               => __( 'Total', 'yith-pre-order-for-woocommerce' ),
					'total_tax'           => __( 'Total tax', 'yith-pre-order-for-woocommerce' ),
					'shipping_total'      => __( 'Order shipping', 'yith-pre-order-for-woocommerce' ),
					'shipping_tax'        => __( 'Order shipping tax', 'yith-pre-order-for-woocommerce' ),
					'discount_total'      => __( 'Discount', 'yith-pre-order-for-woocommerce' ),
					'discount_tax'        => __( 'Discount Tax', 'yith-pre-order-for-woocommerce' ),
					'billing_country'     => __( 'Billing country', 'yith-pre-order-for-woocommerce' ),
					'billing_first_name'  => __( 'Billing first name', 'yith-pre-order-for-woocommerce' ),
					'billing_last_name'   => __( 'Billing last name', 'yith-pre-order-for-woocommerce' ),
					'billing_company'     => __( 'Billing company', 'yith-pre-order-for-woocommerce' ),
					'billing_address_1'   => __( 'Billing address 1', 'yith-pre-order-for-woocommerce' ),
					'billing_address_2'   => __( 'Billing address 2', 'yith-pre-order-for-woocommerce' ),
					'billing_city'        => __( 'Billing city', 'yith-pre-order-for-woocommerce' ),
					'billing_state'       => __( 'Billing state', 'yith-pre-order-for-woocommerce' ),
					'billing_postcode'    => __( 'Billing postcode/ZIP', 'yith-pre-order-for-woocommerce' ),
					'billing_email'       => __( 'Billing email', 'yith-pre-order-for-woocommerce' ),
					'billing_phone'       => __( 'Billing phone', 'yith-pre-order-for-woocommerce' ),
					'shipping_country'    => __( 'Shipping country', 'yith-pre-order-for-woocommerce' ),
					'shipping_first_name' => __( 'Shipping first name', 'yith-pre-order-for-woocommerce' ),
					'shipping_last_name'  => __( 'Shipping last name', 'yith-pre-order-for-woocommerce' ),
					'shipping_company'    => __( 'Shipping company', 'yith-pre-order-for-woocommerce' ),
					'shipping_address_1'  => __( 'Shipping address 1', 'yith-pre-order-for-woocommerce' ),
					'shipping_address_2'  => __( 'Shipping address 2', 'yith-pre-order-for-woocommerce' ),
					'shipping_city'       => __( 'Shipping city', 'yith-pre-order-for-woocommerce' ),
					'shipping_state'      => __( 'Shipping state', 'yith-pre-order-for-woocommerce' ),
					'shipping_postcode'   => __( 'Shipping postcode/ZIP', 'yith-pre-order-for-woocommerce' ),
				)
			);
			$this->export_csv( $pre_orders, $columns );
		}

		/**
		 * CSV export handler.
		 *
		 * @param array $pre_orders Array of items to be exported.
		 * @param array $columns Array of columns for each item.
		 */
		public function export_csv( $pre_orders, $columns ) {

			if ( ! empty( $pre_orders ) ) {

				$formatted_orders = array();

				foreach ( $pre_orders as $pre_order ) {

					foreach ( $columns as $key => $column ) {
						switch ( $key ) {
							case 'status':
								$value = wc_get_order_status_name( $pre_order->get_status() );
								break;
							case 'order_date':
								$value = date_i18n( 'Y-m-d H:i:s', $pre_order->get_date_created()->getTimestamp() );
								break;
							default:
								if ( method_exists( $pre_order, 'get_' . $key ) ) {
									$getter = 'get_' . $key;
									$value  = apply_filters( 'ywpo_export_column_value', $pre_order->$getter(), $pre_order, $key );
								} else {
									$value = $pre_order->get_meta( $key );
									if ( is_array( $value ) ) {
										$value = implode( ' - ', $value );
									}
								}
						}

						$formatted_orders[ $pre_order->get_id() ][] = apply_filters( 'ywpo_export_column_value', $value, $pre_order, $key );
					}
				}

				if ( ! empty( $formatted_orders ) ) {
					$sitename  = sanitize_key( get_bloginfo( 'name' ) );
					$sitename .= ( ! empty( $sitename ) ) ? '-' : '';
					$filename  = $sitename . 'yith-pre-orders-' . gmdate( 'Y-m-d-H-i' ) . '.csv';

					// Add Labels to CSV.
					$formatted_labels[] = array_values( $columns );
					$formatted_orders   = array_merge( $formatted_labels, $formatted_orders );

					header( 'Content-Description: File Transfer' );
					header( 'Content-Disposition: attachment; filename=' . $filename );
					header( 'Content-Type: text/xml; charset=' . apply_filters( 'ywpo_csv_charset', get_option( 'blog_charset' ) ) );

					$df = fopen( 'php://output', 'w' );

					foreach ( $formatted_orders as $row ) {
						fputcsv( $df, $row, ';' );
					}
					// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_read_fclose
					fclose( $df );
				}
			}

			die();
		}

		/**
		 * Prepare pre-order products list table
		 */
		public function list_table_products() {
			global $wc_list_table, $wp_list_table;

			$this->list_table_post_type = 'product';

			require_once YITH_WCPO_PATH . 'includes/admin/class-yith-pre-order-list-table-products.php';
			$wc_list_table = new YITH_Pre_Order_List_Table_Products();

			require_once YITH_WCPO_PATH . 'includes/admin/class-yith-pre-order-products-list.php';
			//phpcs:ignore WordPress.WP.GlobalVariablesOverride.Prohibited
			$wp_list_table = new YITH_Pre_Order_Products_List( array( 'screen' => "edit-$this->list_table_post_type" ), $this->list_table_post_type );

			$this->render_list_table( $wp_list_table );
		}

		/**
		 * Prepare pre-order orders list table
		 */
		public function list_table_orders() {
			if ( OrderUtil::custom_orders_table_usage_is_enabled() ) {
				$screen            = get_current_screen();
				$screen->id        = 'woocommerce_page_wc-orders';
				$screen->post_type = 'shop_order';

				require_once YITH_WCPO_PATH . 'includes/admin/class-yith-pre-order-hpos-orders-list-table.php';

				$orders_table = new YITH_Pre_Order_HPOS_Orders_List_Table();
				$orders_table->setup(
					array(
						'order_type' => 'shop_order',
					)
				);

				if ( $orders_table->current_action() ) {
					$orders_table->handle_bulk_actions();
				}

				// Strip HTTP referer.
				$current_url  = esc_url_raw( wp_unslash( $_SERVER['REQUEST_URI'] ?? '' ) );
				$stripped_url = remove_query_arg( array( '_wp_http_referer', '_wpnonce' ), $current_url );

				if ( $stripped_url !== $current_url ) {
					wp_safe_redirect( $stripped_url );
					exit;
				}

				// Get items and display table.
				$orders_table->prepare_items();
				$orders_table->display();
			} else {
				global $wc_list_table, $wp_list_table;

				$this->list_table_post_type = 'shop_order';

				require_once YITH_WCPO_PATH . 'includes/admin/class-yith-pre-order-list-table-orders.php';
				$wc_list_table = new YITH_Pre_Order_List_Table_Orders();

				require_once YITH_WCPO_PATH . 'includes/admin/class-yith-pre-order-orders-list.php';
				//phpcs:ignore WordPress.WP.GlobalVariablesOverride.Prohibited
				$wp_list_table = new YITH_Pre_Order_Orders_List( array( 'screen' => "edit-$this->list_table_post_type" ), $this->list_table_post_type );

				$this->render_list_table( $wp_list_table );
			}
		}

		/**
		 * Render the list table.
		 *
		 * @param WP_Posts_List_Table $wp_list_table The list table. It could be YITH_Pre_Order_Orders_List
		 * or YITH_Pre_Order_Orders_List.
		 */
		public function render_list_table( $wp_list_table ) {
			$args = array(
				'post_type'     => $this->list_table_post_type,
				'wp_list_table' => $wp_list_table,
			);
			wc_get_template( 'admin/ywpo-list-table.php', $args, '', YITH_WCPO_TEMPLATE_PATH . '/' );
		}

		/**
		 * Handle bulk actions for products.
		 *
		 * @param  string $redirect_to URL to redirect to.
		 * @param  string $action      Action name.
		 * @param  array  $ids         List of ids.
		 *
		 * @return string|WP_Error
		 */
		public function handle_product_bulk_actions( $redirect_to, $action, $ids ) {
			if ( 'ywpo-set-pre-order' === $action || 'ywpo-remove-pre-order' === $action ) {
				$changed = 0;
				$ids     = apply_filters( 'ywpo_products_bulk_action_ids', array_reverse( array_map( 'absint', $ids ) ), $action, 'order' );
				try {
					foreach ( $ids as $id ) {
						$pre_order = ywpo_get_pre_order( $id );
						if ( $pre_order->product->is_type( 'simple' ) ) {
							$pre_order->set_pre_order_status( 'ywpo-set-pre-order' === $action ? 'yes' : 'no' );
							if ( 'ywpo-set-pre-order' === $action && ! $pre_order->get_pre_order_mode() ) {
								$pre_order->set_pre_order_mode( 'manual' );
							}
							$changed++;
						} elseif ( $pre_order->product->is_type( 'variable' ) ) {
							$children = $pre_order->product->get_children();
							if ( empty( $children ) ) {
								continue;
							}
							$count    = count( $children );
							$enabled  = 0;
							$disabled = 0;
							foreach ( $children as $variation_id ) {
								$pre_order_variation = ywpo_get_pre_order( $variation_id );
								if ( 'ywpo-set-pre-order' === $action && ! $pre_order->get_pre_order_mode() ) {
									$pre_order_variation->set_pre_order_status( 'yes' );
									$enabled++;
									if ( ! $pre_order->get_pre_order_mode() ) {
										$pre_order->set_pre_order_mode( 'manual' );
									}
								}
								if ( 'ywpo-remove-pre-order' === $action ) {
									$pre_order_variation->set_pre_order_status( 'no' );
									$disabled++;
								}
								$changed++;
							}
							if ( $enabled > 0 ) {
								update_post_meta( $id, '_ywpo_has_variations', 'yes' );
							}
							if ( metadata_exists( 'post', $id, '_ywpo_has_variations' ) && $count === $disabled ) {
								update_post_meta( $id, '_ywpo_has_variations', 'no' );
							}
						}
					}
					$redirect_to = add_query_arg( $action, count( $ids ), $redirect_to );
				} catch ( Exception $e ) {
					return new WP_Error( $e->getCode(), $e->getMessage() );
				}

				if ( $changed ) {
					$redirect_to = add_query_arg(
						array(
							'post_type'   => 'product',
							'bulk_action' => $action,
							'changed'     => $changed,
							'ids'         => join( ',', $ids ),
						),
						$redirect_to
					);
				}
			}

			return esc_url_raw( $redirect_to );
		}

		/**
		 * Handle bulk actions for taxonomies.
		 */
		public function handle_taxonomy_bulk_actions() {
			global $typenow;
			$post_type      = $typenow;
			$current_screen = get_current_screen();
			$sendback       = '';
			$taxonomy       = '';
			if ( 'edit-product_cat' === $current_screen->id ) {
				$sendback = admin_url( "edit-tags.php?taxonomy=product_cat&post_type=$post_type" );
				$taxonomy = 'product_cat';
			}
			if ( 'edit-product_tag' === $current_screen->id ) {
				$sendback = admin_url( "edit-tags.php?taxonomy=product_tag&post_type=$post_type" );
				$taxonomy = 'product_tag';
			}
			$wp_list_table = _get_list_table( 'WP_Posts_List_Table' );
			$action        = $wp_list_table->current_action();

			if ( ( 'ywpo-set-pre-order' === $action ) || ( 'ywpo-remove-pre-order' === $action ) ) {
				$category_ids = isset( $_POST['delete_tags'] ) && is_array( $_POST['delete_tags'] ) ? array_map( 'sanitize_text_field', wp_unslash( $_POST['delete_tags'] ) ) : false;
				check_admin_referer( 'bulk-tags' );
				foreach ( $category_ids as $category_id ) {
					$args     = array(
						'post_type'   => 'product',
						'numberposts' => - 1,
						'fields'      => 'ids',
						'tax_query'   => array( // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_tax_query
							array(
								'taxonomy' => $taxonomy,
								'field'    => 'term_id',
								'terms'    => $category_id,
								'operator' => 'IN',
							),
						),
					);
					$post_ids = get_posts( $args );
					if ( ! $post_ids ) {
						continue;
					}
					foreach ( $post_ids as $post_id ) {
						$pre_order = ywpo_get_pre_order( $post_id );
						if ( $pre_order->product->is_type( 'simple' ) ) {
							$pre_order->set_pre_order_status( 'ywpo-set-pre-order' === $action ? 'yes' : 'no' );
							if ( 'ywpo-set-pre-order' === $action && ! $pre_order->get_pre_order_mode() ) {
								$pre_order->set_pre_order_mode( 'manual' );
							}
						} elseif ( $pre_order->product->is_type( 'variable' ) && $pre_order->product->has_child() ) {
							$children = $pre_order->product->get_children();
							if ( empty( $children ) ) {
								continue;
							}
							$count    = count( $children );
							$enabled  = 0;
							$disabled = 0;
							foreach ( $children as $variation_id ) {
								$pre_order_variation = ywpo_get_pre_order( $variation_id );
								if ( 'ywpo-set-pre-order' === $action && ! $pre_order->get_pre_order_mode() ) {
									$pre_order_variation->set_pre_order_status( 'yes' );
									$enabled++;
									if ( ! $pre_order->get_pre_order_mode() ) {
										$pre_order->set_pre_order_mode( 'manual' );
									}
								}
								if ( 'ywpo-remove-pre-order' === $action ) {
									$pre_order_variation->set_pre_order_status( 'no' );
									$disabled++;
								}
							}
							if ( $enabled > 0 ) {
								update_post_meta( $post_id, '_ywpo_has_variations', 'yes' );
							}
							if ( metadata_exists( 'post', $post_id, '_ywpo_has_variations' ) && $count === $disabled ) {
								update_post_meta( $post_id, '_ywpo_has_variations', 'no' );
							}
						}
					}
				}
				wp_safe_redirect( $sendback );
				exit();
			}
		}

		/**
		 * Adds the posts per page screen option for the Pre-order Products and Pre-orders list tables.
		 */
		public function add_posts_per_page_screen_option() {
			if ( isset( $_GET['page'] ) && 'yith_wcpo_panel' === $_GET['page'] ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended
				$post_type = 'product';
				if ( isset( $_GET['tab'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended
					if ( 'pre-orders' !== $_GET['tab'] ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended
						$post_type = '';
					}
					if ( isset( $_GET['sub_tab'] ) && 'orders' === $_GET['sub_tab'] ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended
						$post_type = 'shop_order';
					}
				}
				if ( $post_type ) {
					add_screen_option(
						'per_page',
						array(
							'default' => 40,
							'option'  => 'edit_' . $post_type . '_per_page',
						)
					);
				}
			}
		}
	}
}

/**
 * Unique access to instance of YITH_Pre_Order_Admin_Premium class
 *
 * @return YITH_Pre_Order_Admin_Premium
 */
function YITH_Pre_Order_Admin_Premium() { // phpcs:ignore WordPress.NamingConventions.ValidFunctionName.FunctionNameInvalid
	return YITH_Pre_Order_Admin_Premium::get_instance();
}
