<?php
/**
 * This file belongs to the YIT Framework.
 *
 * This source file is subject to the GNU GENERAL PUBLIC LICENSE (GPL 3.0)
 * that is bundled with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * http://www.gnu.org/licenses/gpl-3.0.txt
 *
 * @package YITH\PreOrder\Includes
 * @author YITH <plugins@yithemes.com>
 */

if ( ! defined( 'YITH_WCPO_VERSION' ) ) {
	exit( 'Direct access forbidden.' );
}

if ( ! class_exists( 'YITH_Pre_Order_Stock_Manager' ) ) {
	/**
	 * Class YITH_Pre_Order_Stock_Manager
	 */
	class YITH_Pre_Order_Stock_Manager {

		/**
		 * Main Instance
		 *
		 * @var YITH_Pre_Order_Stock_Manager
		 * @since  1.0
		 */
		protected static $instance;

		/**
		 * Returns single instance of the class
		 *
		 * @return YITH_Pre_Order_Stock_Manager
		 * @since 1.0.0
		 */
		public static function get_instance() {
			if ( is_null( self::$instance ) ) {
				self::$instance = new self();
			}

			return self::$instance;
		}

		/**
		 * Construct
		 *
		 * @since 1.3.0
		 */
		public function __construct() {
			add_filter( 'woocommerce_product_is_in_stock', array( $this, 'allow_outofstock_selling' ), 10, 2 );
			add_filter( 'woocommerce_product_backorders_allowed', array( $this, 'allow_outofstock_selling' ), 10, 2 );
			add_action( 'woocommerce_product_set_stock', array( $this, 'send_out_of_stock_notification' ) );
			add_action( 'woocommerce_variation_set_stock', array( $this, 'send_out_of_stock_notification' ) );
		}

		/**
		 * Allow out of stock pre-order products to be added to the cart when the Automatic Pre-order option is enabled.
		 *
		 * @param bool           $return  Whether the product is in stock or not.
		 * @param WC_Product|int $product The WC_Product object or the product ID.
		 *
		 * @return bool
		 */
		public function allow_outofstock_selling( $return, $product ) {
			if ( ywpo_is_admin() ) {
				return $return;
			}

			$product = wc_get_product( $product );

			// If the Automatic Pre-Order status option is enabled for the product and the stock status is out-of-stock, the product will show the Pre-Order status and can be purchased.
			if ( $product instanceof WC_Product && ( $product->is_type( 'simple' ) || $product->is_type( 'variation' ) ) ) {
				if ( ywpo_automatic_pre_order_check( $product ) ) {
					$return = true;
				}
			}

			return $return;
		}

		/**
		 * Check if a product goes out of stock and send notification.
		 *
		 * @param WC_Product $product The product object.
		 */
		public function send_out_of_stock_notification( $product ) {
			if ( 'yes' !== get_option( 'yith_wcpo_enable_pre_order_auto_outofstock_notification', 'no' ) ) {
				return;
			}

			if ( 'instock' !== $product->get_stock_status( 'edit' ) ) {
				WC()->mailer();
				do_action( 'ywpo_out_of_stock_email', $product );
			}
		}
	}
}

/**
 * Unique access to instance of YITH_Pre_Order_Stock_Manager class
 *
 * @return YITH_Pre_Order_Stock_Manager
 */
function YITH_Pre_Order_Stock_Manager() { // phpcs:ignore WordPress.NamingConventions.ValidFunctionName.FunctionNameInvalid
	return YITH_Pre_Order_Stock_Manager::get_instance();
}
