<?php
/**
 * This file belongs to the YIT Framework.
 *
 * This source file is subject to the GNU GENERAL PUBLIC LICENSE (GPL 3.0)
 * that is bundled with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * http://www.gnu.org/licenses/gpl-3.0.txt
 *
 * @package YITH\PreOrder\Includes
 * @author YITH <plugins@yithemes.com>
 */

if ( ! defined( 'YITH_WCPO_VERSION' ) ) {
	exit( 'Direct access forbidden.' );
}

if ( ! class_exists( 'YITH_Pre_Order_Fee' ) ) {
	/**
	 * Class YITH_Pre_Order_Fee
	 * This class manages the operations concerning the pre-order fees.
	 */
	class YITH_Pre_Order_Fee {

		/**
		 * Main Instance
		 *
		 * @var YITH_Pre_Order_Fee
		 * @since  1.0
		 * @access public
		 */
		protected static $instance;

		/**
		 * Returns single instance of the class
		 *
		 * @return YITH_Pre_Order_Fee
		 * @since 1.0.0
		 */
		public static function get_instance() {
			if ( is_null( self::$instance ) ) {
				self::$instance = new self();
			}

			return self::$instance;
		}

		/**
		 * Construct
		 *
		 * @since  1.0
		 */
		public function __construct() {
			if ( 'no' === get_option( 'yith_wcpo_enable_pre_order', 'yes' ) ) {
				return;
			}
			add_filter( 'woocommerce_cart_calculate_fees', array( $this, 'add_pre_order_fee' ) );
			add_filter( 'woocommerce_get_price_html', array( $this, 'change_price_html' ), 110, 2 );
		}

		/**
		 * Calculate the corresponding fee depending on the Pre-Order items on the Cart.
		 *
		 * @param WC_Cart $cart The current Cart.
		 *
		 * @return WC_Cart
		 */
		public function add_pre_order_fee( $cart ) {
			if ( ! ywpo_cart_contains_pre_order_product() ) {
				return $cart;
			}
			$fee_enabled     = get_option( 'ywpo_fee', 'no' );
			$fee_amount      = apply_filters( 'ywpo_fee_amount', get_option( 'ywpo_fee_amount', 0 ), $cart );
			$fee_include_tax = 'yes' === get_option( 'ywpo_fee_include_tax', 'yes' );
			$fee_taxable     = ! ( 'non-taxable' === get_option( 'ywpo_fee_include_tax', 'yes' ) );

			$fee_totals = array();
			foreach ( $cart->cart_contents as $item ) {
				$product_id           = ! empty( $item['variation_id'] ) ? $item['variation_id'] : $item['product_id'];
				$override_fee_enabled = get_post_meta( $product_id, '_ywpo_override_fee', true );
				$override_fee         = apply_filters( 'ywpo_override_fee_amount', get_post_meta( $product_id, '_ywpo_fee', true ), $product_id, $item );
				$is_pre_order         = YITH_Pre_Order_Utils::is_pre_order_active( $product_id );

				if ( apply_filters( 'ywpo_fee_check_pre_order_product', ! $is_pre_order, $product_id, $item ) ) {
					continue;
				}

				if ( 'yes' === $override_fee_enabled && $override_fee > 0 ) {
					$fee_totals[] = (float) $override_fee * (int) $item['quantity'];
				} elseif ( 'yes' === $fee_enabled && $fee_amount > 0 ) {
					$fee_totals[] = (float) $fee_amount * (int) $item['quantity'];
				}
			}

			$fee_total = apply_filters( 'ywpo_fee_total', array_sum( $fee_totals ), $fee_totals, $fee_amount );
			if ( wc_tax_enabled() && $fee_include_tax ) {
				$taxes     = (float) array_sum( WC_Tax::calc_tax( $fee_total, WC_Tax::get_rates(), true ) );
				$fee_total = apply_filters( 'ywpo_fee_total_without_taxes', $fee_total - $taxes, $fee_total, $taxes );
			}

			if ( $fee_total > 0 ) {
				$fee_name    = apply_filters( 'ywpo_pre_order_fee_name', __( 'Pre-order fee', 'yith-pre-order-for-woocommerce' ), $fee_total, $fee_totals );
				$fee_taxable = apply_filters( 'ywpo_pre_order_fee_taxable', $fee_taxable, $fee_total );

				$cart->add_fee( $fee_name, $fee_total, $fee_taxable );
			}

			return $cart;
		}

		/**
		 * Show the fee amount the pre-order will charge on the product page.
		 *
		 * @param string     $price Price.
		 * @param WC_Product $product Product.
		 *
		 * @return string
		 */
		public function change_price_html( $price, $product ) {
			if ( apply_filters( 'ywpo_fee_restrict_pages', ! is_product(), $product, $price ) ) {
				return $price;
			}

			$fee_label = get_option( 'ywpo_product_page_fee_label', '<br><span style="color: #467484;">+ a fee of {fee}</span>' );
			if ( 'yes' !== get_option( 'ywpo_show_fee_on_product_page', 'no' ) || empty( $fee_label ) ) {
				return $price;
			}

			if ( ! YITH_Pre_Order_Utils::is_pre_order_active( $product ) ) {
				return $price;
			}

			$fee_include_tax = get_option( 'ywpo_fee_include_tax', 'yes' );

			$fee_amount = 0.0;

			$override_fee_enabled = YITH_Pre_Order_Utils::get_override_fee( $product );
			$override_fee         = YITH_Pre_Order_Utils::get_fee( $product );

			if ( 'yes' === $override_fee_enabled && $override_fee ) {
				$fee_amount = (float) $override_fee;
			} else {
				$fee_enabled = get_option( 'ywpo_fee', 'no' );
				$global_fee  = (float) get_option( 'ywpo_fee_amount', 0 );
				if ( 'yes' === $fee_enabled && $global_fee > 0.0 ) {
					$fee_amount = $global_fee;
				}
			}

			if ( $fee_amount > 0.0 ) {
				if ( wc_tax_enabled() ) {
					$tax_display_shop = get_option( 'woocommerce_tax_display_shop' );
					if ( 'no' === $fee_include_tax && 'incl' === $tax_display_shop ) {
						$taxes      = (float) array_sum( WC_Tax::calc_tax( $fee_amount, WC_Tax::get_rates() ) );
						$fee_amount = $fee_amount + $taxes;
					} elseif ( 'yes' === $fee_include_tax && 'excl' === $tax_display_shop ) {
						$taxes      = (float) array_sum( WC_Tax::calc_tax( $fee_amount, WC_Tax::get_rates(), true ) );
						$fee_amount = $fee_amount - $taxes;
					}
				}

				$fee_label = str_replace( '{fee}', wc_price( $fee_amount ), $fee_label );

				$price .= '<span class="ywpo-fee-detail"> ' . nl2br( $fee_label ) . '</span>';
			}

			return $price;

		}
	}
}

/**
 * Unique access to instance of YITH_Pre_Order_Fee
 *
 * @return YITH_Pre_Order_Fee
 */
function YITH_Pre_Order_Fee() { // phpcs:ignore WordPress.NamingConventions.ValidFunctionName.FunctionNameInvalid
	return YITH_Pre_Order_Fee::get_instance();
}
